function result = solTensor(xobs, yobs, zobs, ...
    xmin, ymin, zmin, xmax, ymax, zmax, mode, rho)
    % The potential tensor of gravity
    
    % density :: rho (unit: kg m-3)
    G0 = 6.67428e-11;           % gravitational constant(unit: m3 kg-1 s-2 )
    ix = [xmin; xmax] - xobs;
    jy = [ymin; ymax] - yobs;
    kz = [zmin; zmax] - zobs;

    switch upper(mode)
        case {'G'}
            % Gravitational potential :: U      (unit: N m kg-1)
            % solve :: \nabla_{p} \nabla_{p} int{ 1/r } \mathbf{d}V
            res = zeros(7, 1);
            for i=1:2
                for j=1:2
                    for k=1:2
                        xx = ix(i);yy = jy(j); zz = kz(k);
                        r = sqrt(xx^2 + yy^2 + zz^2);
                        mu_ijk = (-1)^(i+j+k);
                        UXX = -atan2(xx*r, yy*zz);
                        UYY = -atan2(yy*r, xx*zz);
                        UZZ = -atan2(zz*r, xx*yy);
                        UXY = log(zz + r);
                        UXZ = log(yy + r);
                        UYZ = log(xx + r);
                        UZ = -xx*UXZ - yy*UYZ + zz*atan2(xx*yy, zz*r);
                        res(:, 1) = res(:, 1) + mu_ijk * [
                                UXX; ...
                                UYY; ...
                                UZZ; ...
                                UXY; ...
                                UXZ; ...
                                UYZ; ...
                                UZ
                                ];
                    end
                end
            end
            % Second derivative
            result.GG = G0 * rho * ...
                        [res(1, 1), res(4, 1), res(5, 1);
                         res(4, 1), res(2, 1), res(6, 1);
                         res(5, 1), res(6, 1), res(3, 1)];
            % Gravity anomalies
            result.Gz = G0 * rho * res(7, 1);

        case {'M'}
            % solve :: \nabla_{p} \nabla_{p} int{ 1/r } \mathbf{d}V
            res = zeros(6, 1);
            for i=1:2
                for j=1:2
                    for k=1:2
                        xx = ix(i);yy = jy(j); zz = kz(k);
                        r = sqrt(xx^2 + yy^2 + zz^2);
                        mu_ijk = (-1)^(i+j+k);
                        UXX = -atan2(xx*r, yy*zz);
                        UYY = -atan2(yy*r, xx*zz);
                        UZZ = -atan2(zz*r, xx*yy);
                        UXY = log(zz + r);
                        UXZ = log(yy + r);
                        UYZ = log(xx + r);
                        res(:, 1) = res(:, 1) + mu_ijk * [
                                UXX; ...
                                UYY; ...
                                UZZ; ...
                                UXY; ...
                                UXZ; ...
                                UYZ
                                ];
                    end
                end
            end
            % Second derivative
            result =  [res(1, 1), res(4, 1), res(5, 1);
                      res(4, 1), res(2, 1), res(6, 1);
                      res(5, 1), res(6, 1), res(3, 1)];

        case {'M2'}
            % 参考《长方体磁场及其梯度无解析奇点表达式理论研究》
            res = zeros(6, 1);
            for i=1:2
                for j=1:2
                    for k=1:2
                        xx = ix(i);yy = jy(j); zz = kz(k);
                        r = sqrt(xx^2 + yy^2 + zz^2);
                        mu_ijk = (-1)^(i+j+k);
                        UXX = 2 * atan2(xx, (yy + zz + r));
                        UYY = 2 * atan2(yy, (xx + zz + r));
                        UZZ = 2 * atan2(zz, (xx + yy + r));
                        UXY = log(zz + r);
                        UXZ = log(yy + r);
                        UYZ = log(xx + r);
                        res(:, 1) = res(:, 1) + mu_ijk * [
                                UXX; ...
                                UYY; ...
                                UZZ; ...
                                UXY; ...
                                UXZ; ...
                                UYZ
                                ];
                    end
                end
            end
            % Second derivative
            result = [res(1, 1), res(4, 1), res(5, 1);
                      res(4, 1), res(2, 1), res(6, 1);
                      res(5, 1), res(6, 1), res(3, 1)];

        case {'M3'}
            % 参考《均磁化直立长方体磁力三阶梯度张量无解析奇点正演计算公式研究》
            % Ex Ey Ez
            % Fx Fy Fz
            % Gx Gy Gz
            res = zeros(6, 1);
            for i=1:2
                for j=1:2
                    for k=1:2
                        xx = ix(i);yy = jy(j); zz = kz(k);
                        r = sqrt(xx^2 + yy^2 + zz^2);
                        mu_ijk = (-1)^(i+j+k);
                        Ex = -2 * atan2(yy + zz + r, xx);
                        Fy = -2 * atan2(xx + zz + r, yy);
                        Gz =  2 * atan2(zz, xx + yy + r);
                        Fx = log(zz + r);
                        Gx = log(yy + r);
                        Gy = log(xx + r);
                        res(:, 1) = res(:, 1) + mu_ijk * [
                                Ex; ...
                                Fy; ...
                                Gz; ...
                                Fx; ...
                                Gx; ...
                                Gy
                                ];
                    end
                end
            end
            % Second derivative
            result = [res(1, 1), res(4, 1), res(5, 1);
                      res(4, 1), res(2, 1), res(6, 1);
                      res(5, 1), res(6, 1), res(3, 1)];

        case {'MGRAD'}
            % 参考《均磁化直立长方体磁力三阶梯度张量无解析奇点正演计算公式研究》
            % Todo ::
            % Exx Fxx Gxx     1 4 5
            % Exy Fxy Gxy     4 6 
            % Exz Fxz Gxz ==> 5   8
            % Eyy Fyy Gyy     6 2 7
            % Eyz Fyz Gyz       7 9
            % Ezz Fzz Gzz     8 9 3
            res = zeros(9, 1);
            for i=1:2
                for j=1:2
                    for k=1:2
                        r = sqrt(ix(i)^2+jy(j)^2+kz(k)^2);
                        mu_ijk = (-1)^(i+j+k);
                        EXX = (ix(i)^2 - r*(jy(j)+kz(k)+r)) / ((jy(j)+r)*(kz(k)+r));
                        FYY = (jy(j)^2 - r*(ix(i)+kz(k)+r)) / ((ix(i)+r)*(kz(k)+r));
                        GZZ = (kz(k)^2 - r*(ix(i)+jy(j)+r)) / ((ix(i)+r)*(jy(j)+r));
                        FXX = ix(i) / (kz(k)+r);
                        GXX = ix(i) / (jy(j)+r);
                        FXY = jy(j) / (kz(k)+r);
                        GYY = jy(j) / (ix(i)+r);
                        GXZ = kz(k) / (jy(j)+r);
                        GYZ = kz(k) / (ix(i)+r);
                        
                        res(:, 1) = res(:, 1) + mu_ijk * [
                                EXX; ...
                                FYY; ...
                                GZZ; ...
                                FXX; ...
                                GXX; ...
                                FXY; ...
                                GYY; ...
                                GXZ; ...
                                GYZ
                                ];
                    end
                end
            end
            % Second derivative
            result = [res(1, 1), res(4, 1), res(5, 1);
                      res(4, 1), res(6, 1),      1   ;
                      res(5, 1),      1   , res(8, 1);
                      res(6, 1), res(2, 1), res(7, 1);
                           1   , res(7, 1), res(9, 1);
                      res(8, 1), res(9, 1), res(3, 1)];
        otherwise
            ...
    end

end

