
### Code used for the manuscript
# "Taphonomic controls on benthic foraminiferal community from an Antarctic Fjord (Edisto Inlet, Victoria Land)"
# Galli G. et al.


##### Library ####

library(ggplot2)
library(ggpubr)
library(tidypaleo)
library(tidyverse)
library(mgcv)
library(vegan)
library(FactoMineR)
library(factoextra)
library(ggsci)
library(ggrepel)
library(readxl)
library(ggpubr)
library(scales)

##### Data import ####
###Data in the supplementary "Foraminifera_counts.xlsx" and "Geochemical_Data.xlsx"
#Change the directory accordingly

data_tan <- read_xlsx("Foraminifera_counts.xlsx", sheet=1) #fossil
data_bio <- read_xlsx("Foraminifera_counts.xlsx", sheet=2) #Living
data_bio.comp <- read_xlsx("Foraminifera_counts.xlsx", sheet=3) #composite to compare with living
eco_trait <- read_xlsx("Foraminifera_counts.xlsx", sheet=4) #Used for aggl. vs. calcareous
chronoPb <- read_xlsx("Geochemical_Data.xlsx", sheet=1) #For age-depth modeling
geochem <- read_xlsx("Geochemical_Data.xlsx", sheet=2) #For geochemical properties


##### Age-Depth models ####
#Pb210 excess activity and geochemical analysis

chronoPb.cleaned <- chronoPb[-1,-c(2,3,5,6)]
chronoPb.cleaned$MUC <- factor(chronoPb.cleaned$MUC,
                               levels=c('34','24','178','180','170'))


##### Figure S1 #####
chronoTOT <- ggplot(chronoPb.cleaned, aes(x=`Activity of 210Pb`,
                                          y=`Mid depth (cm)`))+
  geom_point()+
  geom_lineh()+
  facet_geochem_wrap(vars(MUC), nrow=1, scale="fixed")+
  geom_errorbarh(aes(xmax=`Activity of 210Pb`+`Dv. Std of Acvtivity`,
                     xmin=`Activity of 210Pb`-`Dv. Std of Acvtivity`))+
  coord_cartesian(expand=T)+
  xlab("Pb activity")+
  ylab("Depth (cm)")+
  scale_y_reverse()+
  geom_hline(yintercept=0, lty=2)+
  theme_paleo()+
  theme(axis.text.x = element_text(angle=-90, vjust=.5))
chronoTOT

##### Figure 2a ####
chronoTOT.years <- ggplot(chronoPb.cleaned, aes(x=`years CE`,
                                                y=`Mid depth (cm)`,
                                                colour = MUC,
                                                fill=MUC))+
  geom_point(alpha=.8,size=3, shape=21)+
  geom_hline(yintercept=5, lty=2)+
  geom_linerange(aes(
    x = 1899 , ymin=Inf, ymax=5),col="brown",lty=2)+
  geom_linerange(aes(
    x = 1996 , ymin=Inf, ymax=5),col="darkgreen",lty=2)+
  geom_linerange(aes(
    x = 1962 , ymin=Inf, ymax=5),col="grey20",lty=2)+
  geom_linerange(aes(
    x = 1985 , ymin=Inf, ymax=5),col="orange",lty=2)+
  geom_line()+
  geom_label(aes(x=1899, y=35, label="1899"), col="brown",fill="white",
             angle = 0)+
  geom_label(aes(x=1996, y=25, label="1996"), col="darkgreen",fill="white",
             angle = 0)+
  geom_label(aes(x=1962, y=45, label="1962"), col="grey20",fill="white",
             angle = 0)+
  geom_label(aes(x=1985, y=35, label="1985"), col="orange",fill="white",
             angle = 0)+
  geom_text(aes(x=1800, y=3.5, label="5-cm depth"), 
            col="black")+
  ylab("Depth (cm)")+
  xlab("Years CE")+
  scale_fill_manual(values = c("black",
                               "darkgreen",
                               "orange","brown",
                               "darkgrey"),
                    aesthetics = c("color","fill"),
                    name="Site")+
  scale_y_reverse(breaks=seq(0,50,10))+
  scale_x_continuous(breaks=seq(1700,2000,50))+
  theme_paleo()+
  theme(legend.position="bottom")

chronoTOT.years


#### Geochemical data + Planktif Foraminifera ####


colnames(geochem) <- c("Site","Intervals", "Depth","Sand", 
                     "MS", "Dry density", "Water", "Eh", 
                     "OC", "Planktic foraminifera") #renamed for usability

geochem_corr <- round(cor(geochem[,-c(1,2,9)]),2) #pairwise correlations

p.mat <- ggcorrplot::cor_pmat(geochem[,-c(1,2,9)]) #p-values of the correlations


##### Fig. 2c ####



geochem_corr_p <- ggcorrplot::ggcorrplot(geochem_corr, p.mat = p.mat,
                                     colors = c("cadetblue",
                                                "white",
                                                "darkred"),
                                     lab=T,type="lower", outline.color = "black",
                                     insig = "blank",pch.col = "white")+
  theme(legend.position = "none")

geochem_corr_p

#### Fig. 2b ####
geochem.l <- geochem %>%
  pivot_longer(-c(Site,Depth,Intervals), names_to="Param",values_to="Value") %>%
  filter(Param!="Water Content(%)")

geochem.l$Site <- factor(Susci.l$Site,
                        levels=c("34","24","178","180","170"))
geochem.l$Param <- factor(Susci.l$Param,
                        levels=c("Dry density","MS","Eh","OC",
                                 "Sand","Planktic foraminifera"))


geochem.p <- ggplot(Susci.l, aes(y=Depth, x=Value, fill=Site))+
  facet_geochem_wrap(vars(Param), scales = "free_x", nrow=3,
             units=c(Water="%", `Dry density`= "g cm -3" , 
                     MS="SI x 10 -6",
                     Sand="% g/g",`Clast > 1mm`="n°/g", `Tubular fragment`="n°/g", `Planktic foraminifera`="n°/g",
                     Eh="mV", OC="%"))+
  geom_lineh(lty=1, linewidth=.7,
            aes(col=Site))+
  geom_point(size=2.5,shape=21)+
  xlab("")+
  ylab("Depth (cm)")+
  ylim(5,0)+
  coord_cartesian(expand = T)+
  scale_fill_manual(values = c("black",
                               "darkgreen",
                               "orange","brown",
                               "darkgrey"),
                    aesthetics = c("color","fill"))+
  theme_paleo()+
  theme(legend.position="bottom",
        panel.grid.minor.x  = element_blank(),
        panel.grid.major.y = element_blank(),
        axis.text.x = element_text(angle=90,vjust=0.5))
geochem.p





##### Calculating Relative abundances for assemblages ####
#For each dataset we are calculating: % (rb), n°/cm^3 (d) and % of eco_traits


#Relative Abundances

data_tan.rb <- data_tan %>%
  mutate(Total=rowSums(select(.,-c(1:5,60))))
data_tan.rb[,c(6:60)] <-  data_tan.rb[,c(6:60)]/data_tan.rb$Total*100

data_bio.rb <- data_bio %>%
  mutate(Total=rowSums(select(.,-c(1:3))))
data_bio.rb[,-c(1,2,3)] <-  data_bio.rb[,-c(1,2,3)]/data_bio.rb$Total*100
  
data_bio.comp.rb <- data_bio.comp %>%
  mutate(Total=rowSums(select(.,-c(1,2))))
data_bio.comp.rb[,-c(1,2)] <-  data_bio.comp.rb[,-c(1,2)]/data_bio.comp.rb$Total*100           
  
#Densities

sf <- 50/69.3978 #area in cm-3

data_tan.d <- data_tan %>%
  mutate(Total=rowSums(select(.,-c(1:5,61))))
data_tan.d[,c(6:60)] <-  (data_tan.rb[,c(6:60)]/data_tan.d$Split)/sf
data_tan.d$Total <- data_tan.d$Total/data_tan.d$Split/sf

data_bio.d <- data_bio %>%
  mutate(Total=rowSums(select(.,-c(1,2,3))))

data_bio.d[,-c(1,2,3)] <-  data_bio.d[,-c(1,2,3)]/sf

data_bio.comp.d <- data_bio.comp %>%
  mutate(Total=rowSums(select(.,-c(1,2,3))))
data_bio.comp.d[,-c(1,2)] <-  data_bio.comp.d[,-c(1,2)]/sf         
data_bio.comp.d$Total <- data_bio.comp.d$Total/sf

#Agglutinated vs Calcareous

data_bio.eco.comp.rb <- data_bio.comp %>%
  pivot_longer(-c(Site, Depth), names_to = "Species", values_to = "Count") %>%
  left_join(eco_trait, by = "Species") %>%
  group_by(Site, Depth, `Test material`) %>%
  mutate(TestMaterialTotal = sum(Count, na.rm = TRUE)) %>%
  ungroup() %>%
  select(-c(Count,Species)) %>%
  group_by(Site, Depth,`Test material`) %>% 
  slice_head() %>%
  pivot_wider(names_from = `Test material`, values_from = TestMaterialTotal) %>%
  mutate(Total=Agglutinated+Calcareous)

data_bio.eco.comp.rb[,c(3:5)] <- data_bio.eco.comp.rb[,c(3:5)]/data_bio.eco.comp.rb$Total*100

data_bio.eco.comp.rb[is.na(data_bio.eco.comp.rb)] <-  0

summary(data_bio.eco.comp.rb)

head(data_bio.eco.comp.rb)

#Fossil data 

sf <- 50/69.3978

data_tan.eco.comp.rb <- data_tan[,-c(2,4,5,60,61)] %>%
  pivot_longer(-c(Site, Depth), names_to = "Species", values_to = "Count") %>%
  left_join(eco_trait, by = "Species") %>%
  group_by(Site, Depth, `Test material`) %>%
  mutate(TestMaterialTotal = sum(Count, na.rm = TRUE)) %>%
  ungroup() %>%
  select(-c(Count,Species)) %>%
  group_by(Site, Depth,`Test material`) %>% 
  slice_head() %>%
  pivot_wider(names_from = `Test material`, values_from = TestMaterialTotal) %>%
  mutate(Total=Agglutinated+Calcareous)
  

data_tan.eco.comp.rb[,c(3:5)] <- data_tan.eco.comp.rb[,c(3:5)]/data_tan.eco.comp.rb$Total*100

data_tan.eco.comp.rb[is.na(data_tan.eco.comp.rb)] <-  0

summary(data_tan.eco.comp.rb)

head(data_tan.eco.comp.rb)

## For figure 3 and 4 removing < 20 (living) and < 10 (fossil)

data_bio.rb$Depth <-  as.numeric(data_bio.rb$Depth)

data_bio.rb.20 <- data_bio.rb[,-c(1,2,3)] %>% 
  select_if(~ any(. > 20)) %>%
  select_if(where(~ n_distinct(.) > 2))
data_bio.rb.20 <- data_bio.rb.20 %>% 
  mutate(Others=100-rowSums(.))

plot(data_bio.rb.10$Others) #check of the 20% thresholds

data_tan.rb.10 <- data_tan.rb[,-c(1,2,3,4,5,61,62)] %>% 
  select_if(~ any(. > 10)) %>%
  select_if(where(~ n_distinct(.) > 2)) 

data_tan.rb.10 <- data_tan.rb.10 %>% 
  mutate(Others=100-rowSums(.))
plot(data_tan.rb.10$Others)



##### Fig. 3a ####

data_bio.d$Depth <-  as.numeric(data_bio.d$Depth)
data_bio.d$`Depth Interval` <- data_bio$`Depth Interval`
data_bio.d$Site <- factor(
  data_bio.d$Site, levels=c("34","24","178","180","170")
)

Tot.bio.d.plot <- ggplot(data_bio.d, aes(y=`Depth Interval`, x=Total))+
  geom_colh(col="black",fill="grey70")+
  geom_point()+
  coord_cartesian(expand = T)+
  facet_geochem_wrap(vars(Site),nrow=1,scale="free_x")+
  xlab(expression(paste("Density of individuals (n° ",50 * cm^{-3},")")))+
  ylab("Depth (cm)")+
  scale_y_discrete(limits=rev)+
  theme_paleo()

Tot.bio.d.plot

#### Fig. 3b ####

data_bio.rb.10$Station <- data_bio.rb$Site
data_bio.rb.10$Depth <- data_bio.rb$Depth
data_bio.rb.10$`Depth Interval` <- data_bio$`Depth Interval`
data_bio.rb.l <- data_bio.rb.10%>% 
  pivot_longer(-c(Station,Depth,`Depth Interval`), names_to = "Species",values_to="RelAb")

data_bio.rb.l$Station <-factor(
  data_bio.rb.l$Station, levels=c("34","24","178","180","170")
) 

data_bio.rb.l$Species <- factor(data_bio.rb.l$Species,
    levels=c('Ammovertellina sp.',
             'Nodulina dentaliniformis',
             'Reophax scorpiurus',
             'Pseudobolivina antarctica',
             'Paratrochammina bartrami',
             'Paratrochammina tricamerata',
             'Portatrochammina antarctica',
             'Miliammina arenacea',
             'Pyrgo oblonga',
             'Pyrgo williamsoni',
             'Trifarina angulosa',
             'Globocassidulina biora',
             'Globocassidulina subglobosa',
             'Globocassidulina spp.',
             'Melonis affinis',
             'Others')
)

Composition.bio.plot <- ggplot(data_bio.rb.l, aes(x=RelAb, y=`Depth Interval`, fill=Species))+
  geom_colh(col="black")+
  scale_fill_d3(palette="category20")+
  facet_geochem_wrap(vars(Station), nrow=1, scale="free_x")+
  scale_y_discrete(limits=rev)+
  theme_paleo()+
  coord_cartesian(expand=T)+
  ylab("Depth (cm)")+
  xlab("Relative abundance (%)")+
  scale_x_continuous(breaks=seq(0,100,25))+
  theme_paleo()+
  theme(legend.text=element_text(face="italic"),
    panel.grid.major.y=element_line(color="grey"),
    panel.grid.minor.x=element_blank(),
    panel.grid.minor.y = element_blank(),
    axis.minor.ticks.x.bottom = element_line(linewidth =1),
    axis.text.x = element_text(colour = c("black", NA),
                               angle=-90, vjust= 0.4, hjust=-.1)
  )

Composition.bio.plot

#### Fig. 3 ####


ggarrange(
  Tot.bio.d.plot, Composition.bio.plot, nrow=2,
  labels="auto", common.legend=T, vjust=T, legend = "right"
)

#### Fig. 4a ####

data_tan.d.forc$Site <- factor(
  data_tan.d.forc$Site, levels=c("34","24","178","180","170"))

data_tan.d.forc$`Depth Interval` <- data_tan$`Depth Interval`

data_tan.d.forc$`Depth Interval` <- factor(
  data_tan.d.forc$`Depth Interval`, 
  levels=c("0-1","1-2","2-3","3-4","4-5"))


Tot.tan.d.plot <- ggplot(data_tan.d.forc, aes(y=`Depth Interval`, x=Total))+
  geom_col(position="dodge", col="black", fill="grey70")+
  geom_point(size=1.5)+
  coord_cartesian(expand = T)+
  facet_geochem_wrap(vars(Site),nrow=1,scale="fixed")+
  xlab(expression(paste("Density of individuals (n° ",50 * cm^{-3},")")))+
  ylab("Depth (cm)")+
  scale_y_discrete(limits=rev)+
  theme_paleo()+
  theme(panel.grid.minor.x = element_blank(),
        axis.text.x = element_text(size=8))

Tot.tan.d.plot

#### Fig. 4b ####
data_tan.rb.10$Site <- data_tan.rb$Site

data_tan.rb.10$`Depth Interval` <- data_tan$`Depth Interval`
data_tan.rb.10$Depth<- data_tan$Depth

data_tan.rb.10$`Depth Interval` <- factor(
  data_tan.rb.10$`Depth Interval`, 
  levels=c("0-1","1-2","2-3","3-4","4-5"))

data_tan.rb.l <- data_tan.rb.10%>% 
  pivot_longer(-c(Site,`Depth Interval`, Depth), names_to = "Species",values_to="RelAb")

data_tan.rb.l$Site <-factor(
  data_tan.rb.l$Site, levels=c("34","24","178","180","170")
) 

data_tan.rb.l$Species <- factor(data_tan.rb.l$Species,
                                              levels=c(
                                                "Adercotryma glomeratum",
                                                "Cribrostomoides jeffreysii",
                                                "Miliammina arenacea",
                                                "Paratrochammina bartrami",
                                                "Portatrochammina bipolaris",
                                                "Portatrochammina antarctica",
                                                "Psammosphaera fusca",
                                                "Pseudobolivina antarctica",
                                                "Ehrenbergina glabra",
                                                "Epistominella exigua",
                                                "Globocassidulina biora",
                                                "Globocassidulina subglobosa",
                                                "Oolina hexagona",
                                                "Rosalina globularis",
                                                "Trifarina angulosa",
                                                "Others"
                                              )
)

Composition.tan.plot <- ggplot(data_tan.rb.l, 
                               aes(x=RelAb, y=`Depth Interval`, 
                                   fill=Species))+
  geom_col(col="black")+
  scale_fill_d3(palette="category20")+
  facet_geochem_wrap(vars(Site), nrow=1, scale="free_x")+
  scale_y_discrete(limits=rev)+
  theme_paleo()+
  ylab("Depth (cm)")+
  xlab("Relative abundance (%)")+
  scale_x_continuous(breaks=seq(0,100,25))+
  theme_paleo()+
  theme(legend.text=element_text(face="italic"),
              panel.grid.major.y=element_line(color="grey"),
              panel.grid.minor.x=element_blank(),
              panel.grid.minor.y = element_blank(),
              axis.minor.ticks.x.bottom = element_line(linewidth =1),
              axis.text.x = element_text(colour = c("black", NA),
                                         angle=-90, vjust= 0.4, hjust=-.1))

Composition.tan.plot

#### Fig. 4 ####

ggarrange(Tot.tan.d.plot, Composition.tan.plot,
          ncol=1, labels="auto",common.legend = T,
          legend = "right")


#### Fig. 5a ####

data_bio.d.forc <- data_bio.comp.d[,c(1,2,51)]
data_bio.d.forc$Depth <- as.numeric(data_bio.d.forc$Depth)
data_tan.d.forc <- data_tan.d[,c(1,3,62)]


Total.d <- data_bio.d.forc %>%
  left_join(data_tan.d.forc, by=c("Site","Depth"))
colnames(Total.d) <- c("Site","Depth","Living","Dead")
Total.d$`Depth Interval`<- data_tan$`Depth Interval`
Total.d <- Total.d  %>%
  pivot_longer(-c(Site,Depth,`Depth Interval`),
               names_to = "Assemblage", values_to = "Dens")
Total.d$Site <- factor(Total.d$Site,
                          levels=c("34","24","178","180","170"))


Tot.plot <- ggplot(Total.d, aes(y=`Depth Interval`, 
                                x=Dens, fill=Assemblage))+
  geom_col(position="dodge", col="black")+
  facet_wrap(vars(Site),nrow=1)+
  scale_y_discrete(limits=rev)+
  scale_fill_manual(values=c("black","grey"))+
  theme_paleo()+
  ylab("Depth (cm)")+
  xlab(expression(paste(" n°",50 * cm^{-3})))+
  theme(legend.position="right",
        panel.grid.minor.x = element_blank(),
        axis.text.x = element_text(angle=-90,vjust=.5))
Tot.plot

data_bio.eco.comp.rb$`Depth Interval` <- data_tan$`Depth Interval`
Living.c <- data_bio.eco.comp.rb[,-5] %>%
  pivot_longer(-c(Site,Depth,`Depth Interval`),
               names_to = "Test Material", values_to = "RelAb")
Living.c$Depth <- as.numeric(Living.c$Depth)
Living.c$Site <- factor(
  Living.c$Site,
  levels=c("34","24","178","180","170")
)


data_tan.eco.comp.rb$`Depth Interval`<- data_tan$`Depth Interval`
Death.c <-  data_tan.eco.comp.rb[,-5] %>%
  pivot_longer(-c(Site,Depth,`Depth Interval`),
               names_to = "Test Material", values_to = "RelAb")
Death.c$Site <- factor(
  Death.c$Site,
  levels=c("34","24","178","180","170")
)

#### Fig 5b ####

living.c.plot <- ggplot(Living.c, 
                        aes(y=`Depth Interval`, 
                            x=RelAb, fill=`Test Material`))+
  geom_col(col="grey")+
  facet_wrap(vars(Site),nrow=1)+
  scale_y_discrete(limits=rev)+
  viridis::scale_fill_viridis(discrete=T)+
  theme_paleo()+
  ylab("Depth (cm)")+
  xlab("Relative abundance (%)")+
  ggtitle("Living assemblage")+
  theme(axis.minor.ticks.x.bottom = element_line(linewidth =1),
        panel.grid.major.x = element_line(colour = c("white"), 
                                  size = c(0.33, 0.2)),
          axis.minor.ticks.y.left = element_line(linewidth =1),
            axis.text.x = element_text(colour = c("black", NA),
                           angle=-90, vjust= 0.4, hjust=-.1))

living.c.plot

#### Fig. 5c ####

dead.c.plot <- ggplot(Death.c, aes(y=`Depth Interval`, x=RelAb, fill=`Test Material`))+
  geom_col(col="grey")+
  facet_wrap(vars(Site),nrow=1)+
  scale_y_discrete(limits=rev)+
  viridis::scale_fill_viridis(discrete=T)+
  theme_paleo()+
  ylab("Depth (cm)")+
  xlab("Relative abundance (%)")+
  ggtitle("Dead assemblage")+
  theme(axis.minor.ticks.x.bottom = element_line(linewidth =1),
        panel.grid.major.x = element_line(colour = c("white"), 
                                          size = c(0.33, 0.2)),
        axis.minor.ticks.y.left = element_line(linewidth =1),
        axis.text.x = element_text(colour = c("black", NA),
                                   angle=-90, vjust= 0.4, hjust=-.1))

dead.c.plot

#### Fig. 5 ####

ggarrange(
  Tot.plot,
  ggarrange(
    living.c.plot,dead.c.plot,
    common.legend = T,
    labels=c("b","c"), legend = "bottom"
  ),
  nrow=2,
  labels="a",
  legend="right"
)


#### nMDS analysis ####

#living assemblage data analysis

data_bio.comp$Depth <- as.numeric(data_bio.comp.rb$Depth)
data_bio.comp.nmds <- data_bio.comp.rb %>%
  select_if(~ any(. > 5)) %>%
  select_if(where(~ n_distinct(.) > 2)) #filtering
data_bio.comp.nmds$Depth <- data_bio.comp$Depth
data_bio.comp.nmds <- data_bio.comp.nmds %>%
  na.omit()

nmds.bio <- metaMDS(data_bio.comp.nmds[,-c(1,2)],
                    distance="bray") #nmds

list(nmds.bio) #diagnostic of the nmds (stress = 0.15)

nmds.bio.sites <- data.frame(nmds.bio$points) #nmds score sites

nmds.bio.sites$Site <- data_bio.comp.nmds$Site 
nmds.bio.sites$Depth <- data_bio.comp.nmds$Depth

hull.bionmds <- nmds.bio.sites %>% 
  group_by(Site) %>% 
  slice(chull(MDS1, MDS2)) #polygons in the nmds

#fossil assemblage analysis

data_tan.rb.n <- data_tan.rb[,-c(2,4,5,61,62)]
data_tan.rb.n$Depth <- as.numeric(data_tan.rb.n$Depth)
data_tan.comp.nmds <- data_tan.rb.n %>%
  select_if(~ any(. >5)) %>%
  select_if(where(~ n_distinct(.) > 2)) %>%
  na.omit()

data_tan.comp.nmds$Depth <- data_tan$Depth

nmds.tan <- metaMDS(data_tan.comp.nmds[,-c(1,2,23)], 
                    autotransform = T, plot=T,
                    distance="bray")
list(nmds.tan) #diagnostic of the nmds (stress = 0.07)
nmds.tan.sites <- data.frame(nmds.tan$points)
nmds.tan.species <- data.frame(nmds.tan$species)
colnames(data_tan.comp.nmds)[1] <- "Site"
nmds.tan.sites$Site <- data_tan.comp.nmds$Site
nmds.tan.sites$Depth <- data_tan.d$Depth
hull.tannmds <- nmds.tan.sites %>% group_by(Site) %>% 
  slice(chull(MDS1, MDS2))

#Living assemblage without site 34

data_bio.comp.rb$Depth <- as.character(data_bio.comp.rb$Depth)

data_bio.comp.rbno34 <- data_bio.comp.rb %>%
  filter(Site != 34) # For tracking the depth labels

data_bio.comp.nmds.no34 <- data_bio.comp.rb %>%
  filter(Site != 34) %>%
  select_if(~ any(. > 5)) %>%
  select_if(where(~ n_distinct(.) > 2))


data_bio.comp.nmds.no34$Depth <- data_bio.comp.rbno34$Depth
data_bio.comp.nmds.no34 <- data_bio.comp.nmds.no34 %>%
  na.omit()

nmds.bio.no34 <- metaMDS(data_bio.comp.nmds.no34[,-c(1,37)], 
                         autotransform = T, 
                         distance="bray")

list(nmds.bio.no34) #diagnostic of the nmds (s=0.16)

nmds.bio.sites.no34 <- data.frame(nmds.bio.no34$points)
nmds.bio.sites.no34$Depth <- data_bio.comp.nmds.no34$Depth
nmds.bio.sites.no34$Site <- data_bio.comp.nmds.no34$Site

hull.bionmds.no34 <- nmds.bio.sites.no34 %>% 
  group_by(Site) %>% 
  slice(chull(MDS1, MDS2))

#Fossil without 34 

data_tan.comp.nmds.no34 <- data_tan.rb.n %>%
  filter(Site != 34) %>%
  {
    first_two <- select(., 1:2)
    rest <- select(., -(1:2)) %>%
      select(where(~ all(. > 5)))
    bind_cols(first_two, rest)
  }

data_tan.comp.nmds.no34 <- data_tan.comp.nmds.no34[,-12]
colnames(data_tan.comp.nmds.no34)[1] <- "Site"


nmds.tan.no34 <- metaMDS(data_tan.comp.nmds.no34[,-c(1,2,22)],
                         distance="bray")
list(nmds.tan.no34) #s=0.06
nmds.tan.sites.no34 <- data.frame(nmds.tan.no34$points)
nmds.tan.sites.no34$Site<- data_tan.comp.nmds.no34$Site
nmds.tan.sites.no34$Depth <- data_tan.comp.nmds.no34$Depth
hull.tannmds.no34 <- nmds.tan.sites.no34 %>% group_by(Site) %>% 
  slice(chull(MDS1, MDS2))


#### Fig. 6a ####

nmds.bio.sites$Site <- factor(nmds.bio.sites$Site, 
                         levels=c(34,24,178,180,170))
nmds.bio.sites$Depth <- factor(nmds.bio.sites$Depth,
                         levels=c("0.5","1.5","2.5","3.5","4.5"))

sites.bio.nmds.plot <- ggplot(nmds.bio.sites, aes(x=MDS1, y=MDS2, 
                                                  shape=as.factor(Depth),
                               fill=as.factor(Site),
                               col=as.factor(Site)))+ 
  geom_hline(yintercept=0, lty=2,col="grey")+
  geom_vline(xintercept=0, lty=2, col="grey")+
  geom_point(size=3.2)+
  geom_polygon(data = hull.bionmds,inherit.aes = F, 
               aes(x=MDS1, y=MDS2, fill=as.factor(Site), 
                   col=as.factor(Site)),
               alpha=.3)+
  scale_shape_manual(values=c(21:25),name="Sample depth (cm)",
                     label=c("0-1","1-2","2-3","3-4","4-5"))+
  scale_fill_manual(name = "Site",
                   values = c("black","darkgreen","orange","brown","darkgrey"),
                    aesthetics = c("color","fill"))+
  labs(x="MDS1", y="MDS2")+
  theme_paleo()+
  theme(axis.title = element_text(face="bold",size=8),
        legend.position="right",
        legend.title = element_text(face="bold"),
        panel.grid=element_blank())

sites.bio.nmds.plot

#### Fig. 6b ####

nmds.tan.sites$Site <- factor(nmds.tan.sites$Site, 
                              levels=c(34,24,178,180,170))
nmds.tan.sites$Depth <- factor(nmds.tan.sites$Depth,
                               levels=c("0.5","1.5","2.5","3.5","4.5"))

sites.tan.nmds.plot <- ggplot(nmds.tan.sites, aes(x=MDS1, y=MDS2, shape=as.factor(Depth),
                                                  fill=as.factor(Site),
                                                  col=as.factor(Site)))+ 
  geom_hline(yintercept=0, lty=2,col="darkgrey")+
  geom_vline(xintercept=0, lty=2, col="darkgrey")+
  geom_point(size=3.2)+
  geom_polygon(data = hull.tannmds,inherit.aes = F, 
               aes(x=MDS1, y=MDS2, fill=as.factor(Site), 
                   col=as.factor(Site)),
               alpha=.3)+
  scale_shape_manual(values=c(21:25),name="Sample depth (cm)",
                     label=c("0-1","1-2","2-3","3-4","4-5"))+
  scale_fill_manual(name = "Site",
                    values = c("black","darkgreen","orange","brown","darkgrey"),
                    aesthetics = c("color","fill"))+
  labs(x="MDS1", y="MDS2")+
  theme_paleo()+
  theme(axis.title = element_text(face="bold",size=8),
        legend.position="right",
        legend.title = element_text(face="bold"),
        panel.grid=element_blank())

sites.tan.nmds.plot

#### Fig. 6 ####

ggarrange(sites.bio.nmds.plot, sites.tan.nmds.plot,
          common.legend = T, legend = "bottom", labels="auto")

#living without 34 

data_bio.comp.rb$Depth <- as.character(data_bio.comp.rb$Depth)

data_bio.comp.rbno34 <- data_bio.comp.rb %>%
  filter(Site != 34) # For tracking the depth labels

data_bio.comp.nmds.no34 <- data_bio.comp.rb %>%
  filter(Site != 34) %>%
  select_if(~ any(. > 5)) %>%
  select_if(where(~ n_distinct(.) > 2))

data_bio.comp.nmds.no34$Depth <- data_bio.comp.rbno34$Depth
data_bio.comp.nmds.no34 <- data_bio.comp.nmds.no34 %>%
  na.omit()
nmds.bio.no34 <- metaMDS(data_bio.comp.nmds.no34[,-c(1,37)], 
                         autotransform = T, 
                         distance="bray")

list(nmds.bio.no34) #diagnostic of the nmds (s=0.16)

nmds.bio.sites.no34 <- data.frame(nmds.bio.no34$points)
nmds.bio.species.no34 <- data.frame(nmds.bio.no34$species)
nmds.bio.sites.no34$Depth <- data_bio.comp.nmds.no34$Depth
nmds.bio.sites.no34$Site <- data_bio.comp.nmds.no34$Site

hull.bionmds.no34 <- nmds.bio.sites.no34 %>% 
  group_by(Site) %>% 
  slice(chull(MDS1, MDS2))


#fossil without 34

data_tan.comp.nmds.no34 <- data_tan.rb.n %>%
  filter(Site != 34) %>%
  {
    first_two <- select(., 1:2)
    rest <- select(., -(1:2)) %>%
      select(where(~ all(. > 5)))
    bind_cols(first_two, rest)
  }

data_tan.comp.nmds.no34 <- data_tan.comp.nmds.no34[,-12]
colnames(data_tan.comp.nmds.no34)[1] <- "Site"

nmds.bio.no34 <- metaMDS(data_bio.comp.nmds.no34[,-c(1,37)], 
                         autotransform = T, 
                         distance="bray")


### Fig. 7a ####

nmds.bio.sites.no34$Site <- factor(nmds.bio.sites.no34$Site, 
                                 levels=c(24,178,180,170))
nmds.bio.sites.no34$Depth <- factor(nmds.bio.sites.no34$Depth,
                               levels=c("0.5","1.5","2.5","3.5","4.5"))

sites.bio.nmds.plot.no34 <- ggplot(nmds.bio.sites.no34, aes(x=MDS1, y=MDS2, shape=as.factor(Depth),
                                                  fill=as.factor(Site),col=as.factor(Site)))+ 
  geom_hline(yintercept=0, lty=2,col="darkgrey")+
  geom_vline(xintercept=0, lty=2, col="darkgrey")+

  geom_point(size=3.2)+
  geom_polygon(data = hull.bionmds.no34,inherit.aes = F, 
               aes(x=MDS1, y=MDS2, fill=as.factor(Site), col=as.factor(Site)),
               alpha=.3)+#convexhulls
  scale_shape_manual(values=c(21:25),name="Sample depth (cm)",
                     label=c("0-1","1-2","2-3","3-4","4-5"))+
  scale_fill_manual(values = c("darkgreen","orange","brown","darkgrey"),
                    aesthetics = c("color","fill"),
                    name="Location")+
  labs(x="MDS1", y="MDS2")+
  theme_paleo()+
  theme(axis.title = element_text(face="bold",size=8),
        legend.position="right",
        legend.title = element_text(face="bold"),
        panel.grid=element_blank())

sites.bio.nmds.plot.no34
 
### Fig. 7b #### 

nmds.tan.sites.no34$Site <- factor(nmds.tan.sites.no34$Site, 
levels=c(24,178,180,170))

sites.tan.nmds.plot.no34 <- ggplot(nmds.tan.sites.no34, 
                                   aes(x=MDS1, y=MDS2, 
                                       shape=as.factor(Depth),
                                                  fill=as.factor(Site),
                                       col=as.factor(Site)))+ 
  geom_hline(yintercept=0, lty=2,col="darkgrey")+
  geom_vline(xintercept=0, lty=2, col="darkgrey")+
  geom_point(size=3.2)+
  geom_polygon(data = hull.tannmds.no34,inherit.aes = F, 
               aes(x=MDS1, y=MDS2, fill=as.factor(Site), col=as.factor(Site)),
               alpha=.3)+#convexhulls
  scale_shape_manual(values=c(21:25),name="Sample depth (cm)",
                     label=c("0-1","1-2","2-3","3-4","4-5"))+
  scale_fill_manual(values = c("darkgreen","orange","brown","darkgrey"),
                    aesthetics = c("color","fill"),
                    name="Location")+
  labs(x="MDS1", y="MDS2")+
  theme_paleo()+
  theme(axis.title = element_text(face="bold",size=8),
        legend.position="right",
        legend.title = element_text(face="bold"),
        panel.grid=element_blank())

sites.tan.nmds.plot.no34

#### Fig. 7 ####

ggarrange(sites.bio.nmds.plot.no34,sites.tan.nmds.plot.no34,
          common.legend = T, legend="right")
 
###plot side by side (top with 34, bottom without)

ggarrange(sites.bio.nmds.plot.no34, sites.tan.nmds.plot.no34, ncol=2,
          common.legend = T, labels="auto", legend="bottom")



#### Fig. 9 - Comparison with other paleocommunities ####

#Data wrangling

data_tan.rb.comparison <- data_tan.rb.n %>%
  filter(Site %in% c("24","178")) %>%
  select(-Depth)

data_tan.rb.comp.selected <- data_tan.rb.comparison %>%
  select(Site,`Globocassidulina biora`,`Globocassidulina subglobosa`,`Paratrochammina bartrami`,
         `Portatrochammina antarctica`,`Trifarina angulosa`,`Miliammina arenacea`) %>%
  pivot_longer(-Site, names_to="Species", values_to = "Abundance") %>%
  mutate(Period="Recent")
colnames(data_tan.rb.comp.selected)[1] <- "age"
ggplot(data_tan.rb.comp.selected, aes(y=Species, x=Abundance, 
                                      fill=as.factor(age)))+
  geom_boxplot(outliers=T)+
  xlab("")+
  ylab("Relative abundance (%)")+
  theme_minimal()+
  theme(
    axis.text.y= element_text(face="italic")
  )


#Data TR17-08 and division in periods

TR1708 <- read.csv("TR17-08_RelAbund.csv",header=T,sep=";") #Data TR17_08 (https://doi.org/10.5194/cp-21-1661-2025-supplement)

TR1708_selected <- TR1708 %>%
  select(age,Globocassidulina.biora,Globocassidulina.subglobosa,
         Paratrochammina.bartrami,Portatrochammina.antarctica,Trifarina.angulosa,
         Miliammina.arenacea)
TR1708_3600_1500 <- TR1708_selected %>%
  filter(age >= 1500) %>%
  pivot_longer(-age, names_to="Species", values_to = "Abundance") %>%
  mutate(Period="3600-1500 yrs BP")
TR1708_3600_1500$Abundance <- TR1708_3600_1500$Abundance*100
TR1708_3600_1500$Species <-gsub("\\."," ",TR1708_3600_1500$Species)

TR1708_1500_1100 <- TR1708_selected %>%
  filter(age < 1500) %>%
  filter (age >= 1100) %>%
  pivot_longer(-age, names_to="Species", values_to = "Abundance") %>%
  mutate(Period="1500-1100 yrs BP")
TR1708_1500_1100$Abundance <- TR1708_1500_1100$Abundance*100
TR1708_1500_1100$Species <-gsub("\\."," ",TR1708_1500_1100$Species)

TR1708_1100_700 <- TR1708_selected %>%
  filter(age < 1100) %>%
  filter (age >= 700) %>%
  pivot_longer(-age, names_to="Species", values_to = "Abundance") %>%
  mutate(Period="1100-700 yrs BP")
TR1708_1100_700$Abundance <- TR1708_1100_700$Abundance*100
TR1708_1100_700$Species <-gsub("\\."," ",TR1708_1100_700$Species)

TR17_08_divided <- TR1708_3600_1500 %>%
  union(TR1708_1500_1100) %>%
  union(TR1708_1100_700) %>%
  union(data_tan.rb.comp.selected)
TR17_08_divided$Period <-  factor(TR17_08_divided$Period,
                                   levels=c("3600-1500 yrs BP",
                                            "1500-1100 yrs BP",
                                            "1100-700 yrs BP",
                                            "Recent"))

comparison_p <- ggboxplot(TR17_08_divided, y = "Abundance", alpha=.2,
          add = "median", rug = TRUE, scales = "free", 
          fill = "Period",color="Period",facet.by="Species",palette = "lancet")+
  geom_pwc( #geom for pairwise comparisons 
    aes(group = Period),
    method = "wilcoxon", label = "p.signif", 
    step.increase = .1,
    hide.ns = F, tip.length = .01, 
    label.size = 2.5,
    bracket.shorten = .1) +
  coord_cartesian(expand = T)+
  theme_paleo()+
  xlab("")+
  viridis::scale_fill_viridis(discrete=T, option="H")+
  viridis::scale_color_viridis(discrete=T, option="H")+
  scale_y_continuous(expand = expansion(mult = c(0, 0.1)),
                     breaks = function(x) {
                       b <- pretty(x)   # breaks
                       b[b <= 100]      # only the one <= 100
                     })+
  ylab("Relative abundances (%)")+
  theme(legend.position="bottom",
        strip.text= element_text(face="italic"),
        axis.ticks.x = element_blank(),
        axis.text.x = element_blank(),
        panel.grid = element_blank())

comparison_p
