import numpy as np
import matplotlib.pyplot as plt

import pandas as pd

import datetime as dt

import postprocess

import importlib
importlib.reload(postprocess)


# --- Import data (customize) ---
df = pd.read_csv('data\\micro_aeth.csv')
times = pd.to_datetime(df['Time local (hh:mm:ss)_MA043'])

day_split = np.where(np.diff(times) < np.timedelta64(4, 'ns'))[0][0]
is_day = np.full(times.shape, True, dtype=bool)  # split by day
is_day[0:day_split+1] = False  # get second day of tests
# is_day[day_split:] = False  # get first day of tests

# Filter data for the first day.
times = times[is_day]

device = 'MA043'
ATN = df.loc[is_day][f'IR ATN1_{device}'].to_numpy()
dM = df.loc[is_day][f'IR_DMc_{device}'].to_numpy()

dM[ATN < 3] = np.nan  # remove ATN < 3 data (as uncorrected)
ATN[ATN < 3] = np.nan

M = dM / ((1/6) * 75) * 1e3
# ------------------------------


# Apply different algorithms.
M_h, count_h = postprocess.hagler(M, ATN, 0.1)

err0 = 10
M_s, count_s, err_s = postprocess.sipkens(M, ATN, 1/6, 75, err0)

M_sk, err_sk = postprocess.sipkens_kalman(M, 1/6, 75)


# Plot the results.
plt.figure(figsize=(10,13))

def set_xlim():
    plt.xlim(dt.datetime(2025, 8, 11, 10, 30),
            dt.datetime(2025, 8, 11, 14, 00))

plt.subplot(5, 1, 1)
plt.scatter(times, M, 2, color=[0.8,0.8,0.8], label='Original')
plt.plot(times, M_h, label='Hagler', color='#39BB76')
set_xlim()
plt.ylim([-10, 90])
plt.ylabel('BC')

plt.subplot(5, 1, 2)
plt.scatter(times, M, 2, color=[0.8,0.8,0.8], label='Original')
plt.fill_between(times, M_s * (1 - 2*err_s), M_s * (1 + 2*err_s), 
                 color='#2E6C8E', alpha=0.2, label='Error bounds')
plt.plot(times, M_s, label='Sipkens', color='#2E6C8E', linewidth=1)
set_xlim()
plt.ylim([-10, 90])
plt.ylabel('BC')

plt.subplot(5, 1, 3)
plt.scatter(times, M, 2, color=[0.8,0.8,0.8], label='Original')
plt.fill_between(times, M_sk * (1 - 2*err_sk), M_sk * (1 + 2*err_sk), 
                 color='#62337A', alpha=0.2, label='Error bounds')
plt.plot(times, M_sk, label='Sipkens-Kalman', color="#62337A", linewidth=1)
set_xlim()
plt.ylim([-10, 90])
plt.ylabel('BC')

plt.subplot(5, 1, 4)
plt.plot(times, count_h, label='Hagler', color="#39BB76")
plt.plot(times, count_s, label='Sipkens', color='#2E6C8E')
set_xlim()
plt.ylabel('Count in\naveraged segment')
plt.legend()

# Error plot.
plt.subplot(5, 1, 5)
plt.plot(times, err_s, label='Sipkens', color='#2E6C8E')
plt.plot(times, err_sk, label='Sipkens-Kalman', color='#62337A')
plt.hlines(err0 / 100, xmin=np.nanmin(times), xmax=np.nanmax(times), color='k', linestyle='--')
plt.ylabel('Relative error\n(k = 2), UR')
plt.xlabel('Time')
set_xlim()
plt.legend()

plt.savefig('BC.svg', format='svg')
plt.show()
