
import numpy as np
import matplotlib.pyplot as plt

import postprocess

import importlib
importlib.reload(postprocess)

# --- 0 | GENERATE SYNTHETIC DATA ---
Qa = 75
N = int(5e3)

Dt = 1/6
times = np.arange(0, (N + 1) * Dt, Dt)

np.random.seed(0)
BC0 = 20 * (1 + 1/3 * np.sin(0.01 * times / Dt / np.pi))  # sin input
# BC0 = 80 * np.exp(-0.02 * np.mod(times, 200))  # impulse type input

p = 4.91
sL = 0.  # < ignore as this is for a single device (add back for full uncertainty)
gam = 24.2

noise_std = np.sqrt(
    sL**2 * BC0**2 + 
    6 * p * BC0 / Qa + 
    6 * gam**2 / (Qa**2 * Dt)
)
BC = BC0 + np.random.normal(0, noise_std)

ATN = -20 + 0.003 * np.cumsum(BC)
for ii in range(len(ATN)):
    if ATN[ii] > 70:
        ATN[ii:] = ATN[ii:] - ATN[ii] - 20

# Plot BC and ATN
plt.figure(1)
plt.subplot(2, 1, 1)
plt.plot(times, BC)
plt.title("BC")

plt.subplot(2, 1, 2)
plt.plot(times, ATN)
plt.title("ATN")


# --- 1 | SIGNAL AVERAGING ---

BC_h, count_h = postprocess.hagler(BC, ATN, 0.5)  # adapted version of Hagler's method

err0 = 9  # error (k=2) in percent
BC_s, count_s, err_s = postprocess.sipkens(BC, ATN, Dt, Qa, err0)
BC_sk, err_sk = postprocess.sipkens_kalman(BC, Dt, Qa)


# --- 2 | METRICS OF SMOOTHING PERFORMANCE ---
# Comparison plots.
plt.figure(6, figsize=(10, 10))
plt.subplot(4, 1, 1)
plt.scatter(times, BC, 1, color=[0.8,0.8,0.8], label='original')
plt.plot(times, BC_sk, label='Sipkens-Kalman')
plt.plot(times, BC_s, label='Sipkens')
plt.plot(times, BC_h, label='Hagler')
plt.ylabel('Average BC concentration (ng m$^{-3}$)')
plt.legend()

plt.subplot(4, 1, 2)
plt.plot(times, count_h, '-', label='Hagler')
plt.plot(times, count_s, '-', label='Sipkens')
plt.ylabel('# Pts averaged')
plt.legend()

plt.subplot(4, 1, 3)
plt.scatter(times, BC / BC0 - 1, 1, color=[0.8,0.8,0.8], label='original')
plt.plot(times, BC_sk / BC0 - 1, label='Sipkens-Kalman')
plt.plot(times, BC_s / BC0 - 1, label='Sipkens')
plt.plot(times, BC_h / BC0 - 1, label='Hagler')
plt.axhline(err0 / 100, color='k', linestyle='--')
plt.axhline(-err0 / 100, color='k', linestyle='--')
plt.ylim([-1.5, 1.5])
plt.ylabel('Relative error')
plt.legend()

plt.subplot(4, 1, 4)
plt.scatter(times, BC - BC0, 1, color=[0.8,0.8,0.8], label='original')
plt.plot(times, BC_sk - BC0, label='Sipkens-Kalman')
plt.plot(times, BC_s - BC0, label='Sipkens')
plt.plot(times, BC_h - BC0, label='Hagler')
plt.ylim([-20, 20])
plt.ylabel('Absolute error')
plt.legend()

plt.tight_layout()
plt.show()
