
import numpy as np
import matplotlib.pyplot as plt

# --- Error model parameters ---
gam = 24.2
p = 4.91
sL = 0.1

# --- Experimental settings ---
delT = 1/6  # in minutes
Qa = 75     # mL/min

# --- Limit-of-detection for mass concentration ---
LoD = (-6 * p - np.sqrt(36 * p**2 - 24 * (sL**2 - 1/4) * gam**2 / delT)) / \
      (2 * Qa * (sL**2 - 1/4))

print(f"LoD = {LoD}")

# --- Error functions ---
def err(M):
    return np.sqrt(sL**2 * M**2 + 6*p/Qa * M + 6*gam**2/(delT * Qa**2))

def rel_err(M):
    return 2 * 100 * np.sqrt(sL**2 + 6*p / (M * Qa) + 6*gam**2 / (delT * (M * Qa)**2))

# --- Plot ---
Mvec = np.linspace(0.5, 100, 150)

fig, axs = plt.subplots(2, 1, figsize=(6, 8))

# Relative error plot
axs[0].plot(Mvec, rel_err(Mvec), 'r')
axs[0].axhline(2 * 100 * sL, color='k', linestyle='--')  # inter-device limit
axs[0].axhline(100, color=(0.5, 0.5, 0.5), linestyle='--')  # ~LoD
axs[0].set_yscale('log')
axs[0].set_ylim([10, 300])
axs[0].set_ylabel('CoV or relative error (k = 2)')

# Absolute error plot
axs[1].plot(Mvec, err(Mvec), 'r')
axs[1].axhline(np.sqrt(6*gam**2/(delT * Qa**2)), color=(0.5, 0.5, 0.5), linestyle='--')
axs[1].set_ylabel('Absolute error')
axs[1].set_xlabel('M [µg/m³]')

plt.tight_layout()
plt.show()
