%%THis script is to calculate a synthetic OAE experiment using the
%%Rhodamine and underway data from the LOC-01 cruise, September 2023. This
%%is a product of the LOC-NESS project from Woods Hole Oceanographic
%%Institution and is associated with the manuscript submitted to
%%Biogeosciences entitled "A tracer study for the development of in-water
%%monitoring, reporting, and verification (MRV) of ship-based ocean
%%alkalinity enhancement" by A.V. Subhas et al.

clear all
close all
clc

%% Load Dataset 
load LOC01underway.mat

LOC01underway.Dye_bkgd = LOC01underway.rhodamine_ppb>0.5; % Bkgd values are 0, measureable signal values are 1. 

%% Only look at the time when the dispersal is happening (maybe a bit afterwards) 

%Indices for the main dataset 
t0 = 85374; %'02-Sep-2023 12:19:00'
tf_disp = 87646; %'02-Sep-2023 13:34:44'
tf_track = 152500; %'04-Sep-2023 01:36:32'

LOC01underway.telapsed = LOC01underway.datetime_UTC - LOC01underway.datetime_UTC(t0);

%% Calculate DIC, pH from TA and pCO2. Then, add alkalinity enhancement and see how that compares. 

%First, pull out all values where there is both a TA value AND a rhodamine
%reading 
LOC01UnderwayCalcs = LOC01underway(~isnan(LOC01underway.TA_umol_kg)&~isnan(LOC01underway.rhodamine_ppb),:);

%Calculate indices for the intial and final timepoints of the monitoring
%track
[~,tsidx0] = min(abs(LOC01underway.datetime_UTC(t0)-LOC01UnderwayCalcs.datetime_UTC));
[~,tsidxtrack] = min(abs(LOC01underway.datetime_UTC(tf_track)-LOC01UnderwayCalcs.datetime_UTC));

%Also identify where the pCO2 values are 
LOC01UnderwayfCO2 = LOC01underway(~isnan(LOC01underway.fCO2_water_SST),:);

%Interpolate fCO2 to the TA values 
LOC01UnderwayCalcs.fCO2_interp = interp1(LOC01UnderwayfCO2.datetime_UTC,LOC01UnderwayfCO2.fCO2_water_SST,LOC01UnderwayCalcs.datetime_UTC);
LOC01UnderwayCalcs.fCO2_atm_interp = interp1(LOC01UnderwayfCO2.datetime_UTC,LOC01UnderwayfCO2.fCO2_atm_interp,LOC01UnderwayCalcs.datetime_UTC);

%Calculate stats on this interpolated background
bkgdstats = grpstats(LOC01UnderwayCalcs,"Dye_bkgd",["mean","median","std"],"DataVars",["fCO2_interp","TA_umol_kg","TempC_SBE38_hull","Salinity_PSU_SBE45","rhodamine_ppb"]);

TAvsS = fitlm(LOC01UnderwayCalcs.Salinity_PSU_SBE45(tsidx0:tsidxtrack),LOC01UnderwayCalcs.TA_umol_kg(tsidx0:tsidxtrack));
pCO2vsS = fitlm(LOC01UnderwayCalcs.Salinity_PSU_SBE45(tsidx0:tsidxtrack),LOC01UnderwayCalcs.fCO2_interp(tsidx0:tsidxtrack));

TAvsT = fitlm(LOC01UnderwayCalcs.TempC_SBE38_hull(tsidx0:tsidxtrack),LOC01UnderwayCalcs.TA_umol_kg(tsidx0:tsidxtrack));
pCO2vsT = fitlm(LOC01UnderwayCalcs.TempC_SBE38_hull(tsidx0:tsidxtrack),LOC01UnderwayCalcs.fCO2_interp(tsidx0:tsidxtrack));

dcarbdT = derivnum('T',LOC01UnderwayCalcs.fCO2_interp,LOC01UnderwayCalcs.TA_umol_kg,4,1,LOC01UnderwayCalcs.Salinity_PSU_SBE45,LOC01UnderwayCalcs.TempC_SBE38_hull,25,1,0,15,1,0,0,1,4,1,1,1);

%% Now, only pull out the max values every hour to calculate the "max" CDR potential 

%Create a timetable to pull out hourly max values just using time (column 3) and
%rhodamine concentration (column 15)
LOC01test = table2timetable(LOC01UnderwayCalcs(:,[3 15])); 
LOC01maxppbhr = retime(LOC01test,'hourly','max');

maxidxs = ismember(LOC01UnderwayCalcs.rhodamine_ppb,LOC01maxppbhr.rhodamine_ppb);
LOC01MaxCalcs = LOC01UnderwayCalcs(maxidxs,:);
LOC01MaxCalcs = LOC01MaxCalcs(4:end,:); %Exclude the first few timepoints that do not have any measured dye fluorescence 

%Calculate the background concentrations of the carbonate system at these
Abkgd = CO2SYS(LOC01MaxCalcs.TA_umol_kg,LOC01MaxCalcs.fCO2_interp,1,4,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1);

LOC01MaxCalcs.DICbkgd = Abkgd(:,2);
LOC01MaxCalcs.fCO2bkgd = Abkgd(:,4);
LOC01MaxCalcs.pHbkgd = Abkgd(:,3);
LOC01MaxCalcs.CO2aqbkgd = Abkgd(:,8);
LOC01MaxCalcs.Rbkgd = Abkgd(:,16); 

%Calculate the interpolated "dynamic" background value only using the
%out-of-patch data defined above. Subsequently these calculations use a "2"
%afterwards to distinguish from the "true" baseline calculations 
LOC01UnderwayBkgd = LOC01UnderwayCalcs(LOC01UnderwayCalcs.Dye_bkgd==0,:);
LOC01MaxCalcs.TA_bkgdinterp = interp1(LOC01UnderwayBkgd.datetime_UTC,LOC01UnderwayBkgd.TA_umol_kg,LOC01MaxCalcs.datetime_UTC);
LOC01MaxCalcs.fCO2_bkgdinterp = interp1(LOC01UnderwayBkgd.datetime_UTC,LOC01UnderwayBkgd.fCO2_interp,LOC01MaxCalcs.datetime_UTC);

TAinterpstats = [mean(LOC01MaxCalcs.TA_umol_kg-LOC01MaxCalcs.TA_bkgdinterp,"omitnan") std(LOC01MaxCalcs.TA_umol_kg-LOC01MaxCalcs.TA_bkgdinterp, "omitnan") 
                    mean(LOC01MaxCalcs.TA_umol_kg-bkgdstats.mean_TA_umol_kg(1), "omitnan") std(LOC01MaxCalcs.TA_umol_kg-bkgdstats.mean_TA_umol_kg(1), "omitnan")];
fCO2interpstats = [mean(LOC01MaxCalcs.fCO2_interp-LOC01MaxCalcs.fCO2_bkgdinterp, "omitnan") std(LOC01MaxCalcs.fCO2_interp-LOC01MaxCalcs.fCO2_bkgdinterp, "omitnan")
                    mean(LOC01MaxCalcs.fCO2_interp-bkgdstats.mean_fCO2_interp(1), "omitnan") std(LOC01MaxCalcs.fCO2_interp-bkgdstats.mean_fCO2_interp(1), "omitnan")];

Abkgd2 = CO2SYS(LOC01MaxCalcs.TA_bkgdinterp,LOC01MaxCalcs.fCO2_bkgdinterp,1,4,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1);

LOC01MaxCalcs.DICbkgd2 = Abkgd2(:,2);
LOC01MaxCalcs.fCO2bkgd2 = Abkgd2(:,4);
LOC01MaxCalcs.pHbkgd2 = Abkgd2(:,3);

%Calculate the OAE with 20 tonnes of NaOH (8.92 umol/kg per ppb)
LOC01MaxCalcs.dTA = LOC01MaxCalcs.rhodamine_ppb.*8.92; 
LOC01MaxCalcs.TAoae = LOC01MaxCalcs.TA_umol_kg + LOC01MaxCalcs.dTA;
LOC01MaxCalcs.TAoae2 = LOC01MaxCalcs.TA_bkgdinterp + LOC01MaxCalcs.dTA;

%Calculate the enhanced-only CO2 system using the enhanced TA and the
%calculated DIC
%True baseline
Aoae = CO2SYS(LOC01MaxCalcs.TAoae,LOC01MaxCalcs.DICbkgd,1,2,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1);

LOC01MaxCalcs.DICoae = Aoae(:,2);
LOC01MaxCalcs.fCO2oae = Aoae(:,4);
LOC01MaxCalcs.pHoae = Aoae(:,3);

%Interpolated dynamic baseline 
Aoae2 = CO2SYS(LOC01MaxCalcs.TAoae2,LOC01MaxCalcs.DICbkgd2,1,2,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1);

LOC01MaxCalcs.DICoae2 = Aoae2(:,2);
LOC01MaxCalcs.fCO2oae2 = Aoae2(:,4);
LOC01MaxCalcs.pHoae2 = Aoae2(:,3);

%% Now do the active gas exchange calculation 

z = 11.4; %Mixed layer depth is 11.4 meters, calculated from the mean penetration depth of the RWT signal (>0.4 ppb from CTDs). Stdev is 2.4 meters. 
k = 2.3784; %This is in units of meters per day, Using measured value of 9.91 cm/hr from Guo et al. paper for Southern New England. 
Sol = Aoae(:,63); %This is K0 (solubility) in units of umol/kg/uatm, multiply by density to convert from per kg to per cubic meter  

tau = LOC01MaxCalcs.DICbkgd./LOC01MaxCalcs.CO2aqbkgd./LOC01MaxCalcs.Rbkgd./k.*z; %CO2 equilibration timescale in days

dt = gradient(datenum(LOC01MaxCalcs.datetime_UTC)); %Timestep for each data point in days 

%Enhancement, dilution only calculations (i.e. no updating of carbonate
%chemistry in real-time) 

dDICoae = dt.*k./z.*Sol.*(LOC01MaxCalcs.fCO2oae-LOC01MaxCalcs.fCO2_interp); %Calculate change in DIC every time step for the null hypothesis, also need to multiply by delta-t...
dDICoae2 = dt.*k./z.*Sol.*(LOC01MaxCalcs.fCO2oae2-LOC01MaxCalcs.fCO2_bkgdinterp); %Calculate change in DIC every time step using only "true" background data

sDICenh = cumsum(dDICoae);
sDICenh2 = cumsum(dDICoae2);

LOC01MaxCalcs.DICoae = LOC01MaxCalcs.DICbkgd-sDICenh;
LOC01MaxCalcs.DICoae2 = LOC01MaxCalcs.DICbkgd-sDICenh2;


%% Gas exchange calculations below 
%Set up vectors for time-stepping calculations 
LOC01MaxCalcs.dDICcdr = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dDICtot = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.DICcdr = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dfCO2cdr = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dpHcdr = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.sDICcdr = zeros(length(LOC01MaxCalcs.fCO2oae),1);

LOC01MaxCalcs.dDICcdr2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dDICtot2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.DICcdr2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dfCO2cdr2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.dpHcdr2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);
LOC01MaxCalcs.sDICcdr2 = zeros(length(LOC01MaxCalcs.fCO2oae),1);

%Set the initial state as the OAE case, but then build in gas exchange on top of that
Acdr = Aoae; 
LOC01MaxCalcs.fCO2cdr = Acdr(:,4); 

Acdr2 = Aoae2;
LOC01MaxCalcs.fCO2cdr2 = Acdr2(:,4);

%Run the loop 
for ii = 1:length(LOC01MaxCalcs.fCO2oae)
    
    if ii == 1
        LOC01MaxCalcs.dDICcdr(ii) = dt(ii).*k./z.*Sol(ii).*(LOC01MaxCalcs.fCO2cdr(ii)-LOC01MaxCalcs.fCO2_interp(ii)); %Calculate change in DIC every time step, also need to multiply by delta-t...
       LOC01MaxCalcs.dDICcdr2(ii) = dt(ii).*k./z.*Sol(ii).*(LOC01MaxCalcs.fCO2cdr(ii)-LOC01MaxCalcs.fCO2_bkgdinterp(ii)); %In this case, use  the interpolated background instead of the "real" background

    else 
       LOC01MaxCalcs.dDICcdr(ii) = dt(ii).*k./z.*Sol(ii).*(LOC01MaxCalcs.fCO2cdr(ii)+LOC01MaxCalcs.dfCO2cdr(ii-1)-LOC01MaxCalcs.fCO2_interp(ii)); %Calculate change in DIC every time step, also need to multiply by delta-t...
        LOC01MaxCalcs.dDICcdr2(ii) = dt(ii).*k./z.*Sol(ii).*(LOC01MaxCalcs.fCO2cdr(ii)+LOC01MaxCalcs.dfCO2cdr2(ii-1)-LOC01MaxCalcs.fCO2_bkgdinterp(ii)); %interpolated background here too 
        
    end 

    LOC01MaxCalcs.sDICcdr = cumsum(LOC01MaxCalcs.dDICcdr); 
    LOC01MaxCalcs.sDICcdr2 = cumsum(LOC01MaxCalcs.dDICcdr2);
    
    LOC01MaxCalcs.DICcdr(ii) = LOC01MaxCalcs.DICbkgd(ii)-LOC01MaxCalcs.sDICcdr(ii); %Do the DIC addition cumulatively
    LOC01MaxCalcs.DICcdr2(ii) = LOC01MaxCalcs.DICbkgd2(ii)-LOC01MaxCalcs.sDICcdr(ii);

    Acdr = CO2SYS(LOC01MaxCalcs.TAoae,LOC01MaxCalcs.DICcdr,1,2,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1); %re-calculate the CO2 system with these new values. 
    Acdr2 = CO2SYS(LOC01MaxCalcs.TAoae2,LOC01MaxCalcs.DICcdr2,1,2,LOC01MaxCalcs.Salinity_PSU_SBE45,LOC01MaxCalcs.TempC_SBE38_hull,nan,3,nan,1,1,0,0,1,9,1,1,1); %re-calculate the CO2 system with these new values.
    
    LOC01MaxCalcs.dfCO2cdr(ii) = Acdr(ii,4)-Aoae(ii,4);
    LOC01MaxCalcs.dfCO2cdr2(ii) = Acdr(ii,4)-Aoae(ii,4);

end

LOC01MaxCalcs.pHcdr = Acdr(:,3);
LOC01MaxCalcs.pHcdr2 = Acdr2(:,3);
LOC01MaxCalcs.fCO2cdr = Acdr(:,4);
LOC01MaxCalcs.fCO2cdr2 = Acdr2(:,4);

%% FIGURES FOR PUBLICATION from the LOC01_SyntheticUptake.m script
close all;

c1 = [0 0 0]; %black
c2 = [0 0 255]./256; %blue
c3 = [255 177 78]./256; %Yellow
c4 = [157 2 215]./256;
sz = 10;

cmap = cmocean('thermal');

%% Salinity-property plot figure, colored by whetehr the point is background (defined by the cutoff above) or if it's a "signal" 
fig1=figure('Position',[0 0 1600 600]);

tiledlayout(1,3)
ax1=nexttile;
colormap(cmocean('oxy'))
scatter(LOC01UnderwayCalcs.Salinity_PSU_SBE45(tsidx0:tsidxtrack),LOC01UnderwayCalcs.TA_umol_kg(tsidx0:tsidxtrack),64,LOC01UnderwayCalcs.Dye_bkgd(tsidx0:tsidxtrack),'filled','markeredgecolor',c1);
xlabel('S')
ylabel('TA')
plot_properties;

ax2=nexttile; 
colormap(cmocean('oxy'))
scatter(LOC01UnderwayCalcs.Salinity_PSU_SBE45(tsidx0:tsidxtrack),LOC01UnderwayCalcs.fCO2_interp(tsidx0:tsidxtrack),64,LOC01UnderwayCalcs.Dye_bkgd(tsidx0:tsidxtrack),'filled','markeredgecolor',c1);
xlabel('S')
ylabel('fCO2')
plot_properties;

ax3=nexttile;
colormap(cmocean('oxy'))
scatter(LOC01UnderwayCalcs.TA_umol_kg(tsidx0:tsidxtrack),LOC01UnderwayCalcs.fCO2_interp(tsidx0:tsidxtrack),64,LOC01UnderwayCalcs.Dye_bkgd(tsidx0:tsidxtrack),'filled','markeredgecolor',c1);
xlabel('TA')
ylabel('fCO2')
plot_properties;

%% Synthetic CDR OAE experiment calculations. 
fig2 = figure('Position',[0 0 2000 1200]);
tiledlayout(2,4)

ax1=nexttile;
plot(LOC01underway.telapsed(t0:tf_track),LOC01underway.TA_umol_kg(t0:tf_track),'ko','MarkerSize',sz,'markerfacecolor',c1);
hold on
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.TA_bkgdinterp,'ko','MarkerSize',sz,'markerfacecolor',[1 1 1])
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.TAoae,'ko','MarkerSize',sz,'markerfacecolor',c2); 
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('Total Alkalinity (umol/kg)')
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])

ax2=nexttile;
plot(LOC01underway.telapsed(t0:tf_track),LOC01underway.fCO2_water_SST(t0:tf_track),'ko','MarkerSize',sz,'markerfacecolor',c1); 
hold on
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.fCO2_bkgdinterp,'ko','MarkerSize',sz,'markerfacecolor',[1 1 1])
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.fCO2oae,'ko','MarkerSize',sz,'markerfacecolor',c2); 
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.fCO2cdr,'ko','MarkerSize',sz,'markerfacecolor',c3);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('fCO2 (uatm)')
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])
ylim([0 500])

ax3=nexttile;
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.pHbkgd,'ko','MarkerSize',sz,'markerfacecolor',c1); 
hold on
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.pHbkgd2,'ko','MarkerSize',sz,'markerfacecolor',[1 1 1])
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.pHoae,'ko','MarkerSize',sz,'markerfacecolor',c2);
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.pHcdr,'ko','MarkerSize',sz,'markerfacecolor',c3);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('pH (uatm)')
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])
ylim([7.9 9.2])

ax4=nexttile;
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.DICbkgd,'ko','MarkerSize',sz,'markerfacecolor',c1); 
hold on
plot(LOC01MaxCalcs.telapsed,LOC01MaxCalcs.DICcdr,'ko','MarkerSize',sz,'markerfacecolor',c3);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('DIC (uatm)')
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])
ylim([1940 1980])

ax5=nexttile;
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.dTA(LOC01MaxCalcs.Dye_bkgd==1),'ko','MarkerSize',sz,'markerfacecolor',c2);
xline(LOC01underway.telapsed(tf_disp))
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])
xlabel('Time Elapsed')
ylabel('dTA')


ax6=nexttile;
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),-LOC01MaxCalcs.fCO2oae(LOC01MaxCalcs.Dye_bkgd==1)+LOC01MaxCalcs.fCO2bkgd(LOC01MaxCalcs.Dye_bkgd==1),'ko','MarkerSize',sz,'markerfacecolor',c2);
hold on
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),-LOC01MaxCalcs.fCO2cdr(LOC01MaxCalcs.Dye_bkgd==1)+LOC01MaxCalcs.fCO2bkgd(LOC01MaxCalcs.Dye_bkgd==1),'ko','MarkerSize',sz,'markerfacecolor',c3);
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.fCO2cdr(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.fCO2oae(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz,'markerfacecolor',c4);
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.fCO2cdr2(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.fCO2oae2(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('dpCO2')
ylim([1 450])
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])

ax7=nexttile; 
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.pHoae(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.pHbkgd(LOC01MaxCalcs.Dye_bkgd==1),'ko','MarkerSize',sz,'markerfacecolor',c2);
hold on
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.pHcdr(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.pHbkgd(LOC01MaxCalcs.Dye_bkgd==1),'ko','MarkerSize',sz,'markerfacecolor',c3);
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.pHoae(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.pHcdr(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz,'markerfacecolor',c4);
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),LOC01MaxCalcs.pHoae2(LOC01MaxCalcs.Dye_bkgd==1)-LOC01MaxCalcs.pHcdr2(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('dpH')
ylim([0.001 1.4])
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])

ax8=nexttile;
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),-LOC01MaxCalcs.sDICcdr(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz,'markerfacecolor',c4);
hold on
semilogy(LOC01MaxCalcs.telapsed(LOC01MaxCalcs.Dye_bkgd==1),-LOC01MaxCalcs.sDICcdr2(LOC01MaxCalcs.Dye_bkgd==1),'kd','MarkerSize',sz);
xline(LOC01underway.telapsed(tf_disp))
xlabel('Time Elapsed')
ylabel('dDIC')
xlim([LOC01underway.telapsed(t0) LOC01underway.telapsed(tf_track)])
