#! /usr/bin/python3
from datetime import datetime, timedelta, timezone
# import matplotlib.image as mpimg    # Use only as mpimg.imread()
# import colorsys
# import math
# import glob   # to read files in a folder (2020-08, Futaana), use only as glob.glob()
from urllib import request as web
import urllib
# import re
import judge_aurora_ASC2021 as judge_ASC
import argparse
from pathlib import Path
import tempfile
import os
import logging
from mkdelta import mkdelta

#--------#---------#---------#---------#---------#---------#---------#
# --- python3 program ---
# Automatic Aurora identification program for PNG images
#   - v1 by Dennis van Dijk 2016-08: for old ASC (Nikon D700 camera)
#   - v2 by M. Yamauchi 2020-08: competely new
#   - v3 by M. Yamauchi 2021-02-03:  debugging and re-structuring.
#   - v4 by M. Yamauchi 2021-03: filter has changed/output format
#   - v5.R1 by M. Yamauchi 2021-10-13  (make file of last 20 minutes)
#           /img-sample> python3 nowcast5_index.py
#   - v5.R2 by Peje Nilsson 2021-11:  real-time operation and warning started
#           /img-sample> python3 ***.py
#
#
# at aurora.irf.se:
# aurora_detection/img-sample> python3 nowcast_index5.py
#   input: recent data
#   output: popup window
#
# Convert RGB to HLS for each pixel, and examine H, S, L values.
#      note: HLS means Hue, Lightness, Saturation.
# The hue component is approximate spectrum color.
# The lightness component is an easy indicator for the strength of an aurora.
#      note: L values are difference between png and jpg
#      note: L = 1.0 means a white pixel, and maximum limit exists
# Color of aurora in RGB pictures changes with moon and twilight.
# Therefore, fine tuning is needed for H, S, L ranges.
#
#------------------------------------
# File format (From august 2020)
#  https://www.irf.se/alis/allsky/krn/2021/03/13/20/2021-03-13T20.47.00.000KRN_small.jpeg
# urlfolder  = https://www.irf.se/alis/allsky/krn/2021/03/13/20/
# jpgname = '2021-03-13T20.47.00.000KRN_small.jpeg'

default_output_dir = '%Y/%m'
default_output_fname = Path("krn%Y%m%d_ASCindex.csv")

camera = 'Kiruna3'  # for SONY camera since 2020.11
fileperiod = 3      # for SONY camera since 2020.11
totalpixel = 223 * 223 * 3.1416   # 446x446 = minimum size

header_kiruna = ''' Format                 similar to IAGA-2002                         |
 Source of Data         Swedish Institute of Space Physics           |
 Station Name           Kiruna                                       |
 IAGA CODE              KIR                                          |
 Geodetic Lat.          67.83                                        |
 Geodetic Long.         20.42                                        |
 Elevation              400                                          |
 Pixels (for average)   140000 (2500)                                |
 Reported               occupancy of auroral pixels and intensity    |
   Occupancy (% pixels) Diff(diffuse), Arc, Strong, void, cloud      |
   Intensity (average)  <L>, <L*L*L> (for strongest pixels only)     |
 Sensor                 ASC, Sony alpha7s, Nikon Nikkor 8 mm 1:2.8   |
 Digital Sampling       1 minute                                     |
 Integration time       1 sec                                        |
 Data Type              Provisional                                  |
      DATE,    TIME,DOY, Diff,  Arc,Strong, void,cloud,   <L>,    L3
'''

HHMM_STRING_FORMAT = '%Y-%m-%dT%H:%M'


#
# Based on start_time find the latest analyzed time-point
#
def parse_result_file(args, start_time):
    args.logger.debug("time given to parse_result_file: {}".format(start_time))
    outdir = Path(start_time.strftime(str(args.output_dir)))
    outdir.mkdir(parents=True, exist_ok=True)
    outname = outdir / Path(start_time.strftime(str(args.output_fname)))
    args.logger.info('output file name: {} exists {}'.format(outname, outname.is_file()))
    line = ""
    if outname.is_file():
        # Find the last line in file
        last_line = []
        with open(str(outname)) as f:
            for line in f:
                pass
            last_line = line  # save only date & time
            args.logger.info(last_line)
        if len(last_line) == 16 and last_line[0] == '2':
            start_time = datetime.strptime(last_line, "%Y-%m-%d,%H:%M:%S").replace(
                second=0, microsecond=0, tzinfo=timezone.utc)
            args.logger.info(start_time)
            start_time += args.delta    # One delta beyond last analyzed time
            args.logger.info(start_time)
    else:
        args.logger.info("No output file")
        outname = ''

    args.logger.debug("time returned from parse_result_file: {}".format(start_time))
    return (start_time, outname)


default_server = 'https://www.irf.se'
default_server_dir = 'alis/allsky/krn/%Y/%m/%d/%H'
default_allsky_fname = '%Y-%m-%dT%H.%M.%S.000KRN_small.jpeg'

default_old_server = 'https://www2.irf.se'
default_old_server_dir = 'allsky/%Y/%Y%m%d/jpgs'
default_old_allsky_fname = 'KRN%Y%m%dT%H%M%S{EXPO}Q.JPG'
# exponation, most common first
expos = ['E06000', 'E04000', 'E03000', 'E02000', 'E01500', 'E01000', ]

default_local_server = 'file://'
default_local_server_dir = 'goptavalvasc/allsky/krn/%Y/%m/%d/%H'
default_local_allsky_fname = '%Y-%m-%dT%H.%M.%S.000KRN_small.jpeg'

default_old_local_server = 'file://'
default_old_local_server_dir = 'filbunke/observatory/allsky/%Y/%Y%m%d/jpgs'
default_old_local_allsky_fname = 'KRN%Y%m%dT%H%M%S{EXPO}Q.JPG'

default_temppath = Path('/tmp/nowcast')


def retrieve_image_old(args, image_time):
    dir_time = image_time
    if dir_time.hour < 12:              # files for the night is collected in evening date
        dir_time -= timedelta(hours=12)

    if args.local:
        urlfname = dir_time.strftime(
            str(default_old_local_server
                + '/' + default_old_local_server_dir))
        + '/' + image_time.strftime(default_old_local_allsky_fname)
    else:
        urlfname = dir_time.strftime(
            default_old_server
            + '/' + default_old_server_dir)
        + '/' + image_time.strftime(default_old_allsky_fname)
    urlfname = urlfname.replace('{EXPO}', args.expo)

    args.logger.info('{} => '.format(urlfname))
    try:
        if args.temppath.exists() is False:
            args.temppath.mkdir()
        fp = tempfile.NamedTemporaryFile(delete=False, dir=str(args.temppath))
        (image, msg) = web.urlretrieve(urlfname, fp.name)
        image = Path(image)
        args.logger.debug('Retrived image into {}'.format(image))
        args.logger.debug('Server response: {}'.format(msg))
    except urllib.error.HTTPError as e:
        args.logger.debug('The server couldn\'t fulfill the request. %s' % urlfname)
        args.logger.debug('Error code: ', e.code)
        os.remove(fp.name)
        return None
    except urllib.error.URLError as e:
        args.logger.debug('We failed to reach a server.')
        args.logger.debug('Reason: ', e.reason)
        os.remove(fp.name)
        return None
    else:
        # everything is fine        except urllib.error.URLError as e:
        pass
    return image


def retrieve_image_new(args, image_time):
    if args.local:
        urlfname = image_time.strftime(
            str(default_local_server + '/'
                + default_local_server_dir + '/'
                + default_local_allsky_fname))
    else:
        urlfname = image_time.strftime(
            str(args.server + '/'
                + args.server_dir + '/'
                + args.allsky_fname))
    args.logger.info('{} => '.format(urlfname))
    try:
        if args.temppath.exists() is False:
            args.temppath.mkdir()
        fp = tempfile.NamedTemporaryFile(delete=False, dir=str(args.temppath))
        (image, msg) = web.urlretrieve(urlfname, fp.name)
        image = Path(image)
        args.logger.debug('Retrived image into {}'.format(image))
        args.logger.debug('Server response: {}'.format(msg))
    except urllib.error.HTTPError as e:
        args.logger.debug('The server couldn\'t fulfill the request. %s' % urlfname)
        args.logger.debug('Error code: ', e.code)
        os.remove(fp.name)
        return None
    except urllib.error.URLError as e:
        args.logger.debug('We failed to reach a server.')
        args.logger.debug('Reason: ', e.reason)
        os.remove(fp.name)
        return None
    else:
        # everything is fine        except urllib.error.URLError as e:
        pass
    return image


def retrieve_image(args, image_time):
    if image_time < judge_ASC.kirunaASC_epoch1.replace(tzinfo=image_time.tzinfo):
        for expo in expos:
            args.expo = expo
            image = retrieve_image_old(args, image_time)
            if image is not None:
                break
    else:
        image = retrieve_image_new(args, image_time)
        if image is None:
            # some images are nudged 1 second
            image_time += timedelta(seconds=1)
            image = retrieve_image_new(args, image_time)

    return image


def detect_aurora(args, judge, image_time, image, camera):

    H, L, S, R, G, B, totalpixel = judge.jpg_everything_HLS(image, camera)

    moonradius0 = 5
    filter_level = 10021
    #    derive_mask(H,L,S, R,G,B, camera, ,,,,,)
    #       ## filter_level = *0**: maskK = full mask within radius
    #                       = *1**: maskM = only dense part of maskM_core
    #                       = *2**: maskM_full = pre-defined mask without considering moon
    #    aurorafilter(H,L,S, R,G,B, camera, filter_level)
    #       ## filter_level = ***0: HLS for both cameras (Nikon & SONY)
    #                       = ***1: RGB for SONY camera
    #    moonfilter(H,L,S, R,G,B, camera, 0)
    #       ## filter_level = **0* - **4*: moon filter (core part + without/with surrounding)
    #                                       <== default
    #                       = **5* - **9*: + filling boundaries
    #    mooncenter(L, maskM_core, maskM_blight, maskM_full,camera, radius, filter_level)
    #       ## filter_level = **0* or **5*: use maskM_core  <== to find out center
    #                       = **1* or **6*: use maskM_blight
    #                       = **2* or **7*: use maskM_full <== filling 14*raidus distance
    #                                                          from moon
    # --------#---------#---------#---------#---------#---------#---------#
    count_loop = 0  # not used
    (maskW, maskA, maskS,
     cntMaybe, cntYes, cntStrong, cntVoid, cntDark,
     AverageLumS, expL_strong, LexpL_strong, LLL_strong, radiusL, new_radius, flag,
     cntLight, cntCloud_raw, cntMoon0, Xcenter0, Ycenter0, cntInside,
     Xcenter_local, Ycenter_local, cntMoon2, Xcenter2, Ycenter2
     ) = judge.derive_mask(H, L, S, R, G, B, camera, totalpixel,
                           args.writescreen, args.savefile, args.saveimage,
                           moonradius0, filter_level, count_loop)

    ASCindex = {
        'ts': image_time,
        # cntMaybe, # 0
        # cntYes,   # 1
        # cntStrong,
        'diff': (cntMaybe * 100. / totalpixel),
        'arc': (cntYes * 100. / totalpixel),
        'strong': (cntStrong * 100. / totalpixel),
        'void': cntVoid,
        'cloud': (cntCloud_raw * 100. / totalpixel),
        '<L>': (AverageLumS * 100),
        'L3': (LLL_strong * 100),
        # (expL_strong*100),
        # (LexpL_strong*100),
        # cntDark,
        # (cntLight*100./totalpixel),
        # (cntMoon*100./totalpixel),
        # flag,
        # radiusL,
        # new_radius
    }
    # ASCpara = (cntMoon0, Xcenter0, Ycenter0, cntInside,
    #            Xcenter_local, Ycenter_local, cntMoon2, Xcenter2, Ycenter2)

    # if args.writescreen:
    #     cntMoon = max(cntMoon0, cntMoon2)
    #     print('flag={}, url={}'.format(flag,
    #                                    image_time.strftime(str(judge_ASC.NEW_ALLSKY_JPEG_FORMAT))))
    #     print(' (Maybe,Yes,Strong)={0:d}, {1:d}, {2:d}'
    #           ' (Cloud,Moon,light)={3:d}%({4:d}, {5:d}, {6:d}) '
    #           ' <L>, L**3={7:.2f}, {8:.3f}'.format(
    #                 cntMaybe, cntYes, cntStrong,
    #                 int(cntVoid), cntCloud_raw, cntMoon, cntLight,
    #                 AverageLumS, LLL_strong))
    return ASCindex


def main(args=None):
    end_time = args.end_time
    outname_old = ''
    if args.parse is not None and args.parse:
        # parse time from current result file
        (start_time, outname_old) = parse_result_file(args, end_time - args.delta)
    elif args.range is None:
        start_time = args.start_time
    else:
        start_time = end_time - args.range

    delta = args.delta

    args.logger.warning('nowcast:{} -- {}'.format(start_time.strftime(HHMM_STRING_FORMAT),
                                                  end_time.strftime(HHMM_STRING_FORMAT)))

    # Copy/simplifiedversion          judge_ASC.list_kirunaASC
    if end_time < judge_ASC.kirunaASC_epoch1.replace(tzinfo=end_time.tzinfo):
        camera = 'Kiruna'
    elif end_time < judge_ASC.kirunaASC_epoch2.replace(tzinfo=end_time.tzinfo):
        camera = 'Kiruna2'
    else:  # obs_period == 3:
        camera = 'Kiruna3'

    while start_time < end_time:
        image_time = start_time

        image = retrieve_image(args, image_time)
        if image is None:
            start_time += delta
            continue

        args.logger.info(image_time.strftime("%FT%T"))

        ASCindex = detect_aurora(args, judge_ASC, image_time, image, camera)

        if args.savefile:
            args.logger.debug("savefile")
            outdir = Path(image_time.strftime(str(args.output_dir)))
            outdir.mkdir(parents=True, exist_ok=True)
            outname = Path(image_time.strftime(str(args.output_fname)))
            outname = outdir / outname
            if not outname.is_file() or outname != outname_old:
                args.logger.debug("{} = not {} isfile {}".format(not outname.is_file(),
                                                                 outname, outname.is_file()))
                args.logger.debug(" and '{}' != '{}'".format(outname, outname_old))
                # Create a new file or overwrite existing
                fileout = open(str(outname), 'w')
                args.logger.info('Saving to {}'.format(outname))
                # fileout.write(header_kiruna_old.format(camera))
                fileout.write(header_kiruna)
                outname_old = outname
                # After writing header, the actual content is obtained line by line in the filelist
            else:
                # append to existing file
                fileout = open(str(outname), 'a')
                args.logger.info('Appending to {}'.format(outname))

            fileout.write('{0:s},'      # DATE TIME DOY
                          '{1:>5.1f},'    # Diff
                          '{2:>5.1f},'    # Arc
                          '{3:>6.2f},'    # Strong
                          '{4:>5.1f},'    # void
                          '{5:>5.1f},'    # cloud
                          '{6:>6.1f},'    # <L>_2500
                          '{7:>6.1f}'     # L3_2500
                          '\n'
                          .format(image_time.strftime("%F,%T,%j"),
                                  ASCindex['diff'],   # Diff
                                  ASCindex['arc'],   # Arc
                                  ASCindex['strong'],   # Strong
                                  ASCindex['void'],   # void
                                  ASCindex['cloud'],   # cloud
                                  ASCindex['<L>'],   # <L>2500
                                  ASCindex['L3'],   # L3_2500
                                  ))
            fileout.close()
        else:
            args.logger.info('Not saving to file')

        start_time += delta  # advance to next timestep

        # Remove downloaded image
        args.logger.info('unlinking: {}'.format(image))
        image.unlink()

    if 'fileout' in locals() and not fileout.closed:
        fileout.close()


if __name__ == '__main__':
    logging.basicConfig(format='%(asctime)s - %(levelname)s: %(message)s')
    logger = logging.getLogger(__name__)
    now = datetime.utcnow().replace(second=0, microsecond=0, tzinfo=timezone.utc)
    default_delta = timedelta(minutes=1)
    parser = argparse.ArgumentParser()
    parser.add_argument('--delta',
                        type=mkdelta,
                        help="Process images every DELTA (default: 1m)",
                        default=default_delta)
    parser.add_argument(
        "-log",
        "--log",
        default="warning",
        help=(
            "Provide logging level. "
            "Example --log debug', default='warning'"
        ),
    )

    group = parser.add_mutually_exclusive_group()
    group.add_argument('--range',
                       type=mkdelta, help="Process the last RANGE")
    group.add_argument('--parse', help="Parse start time from possible existing result file",
                       action='store_true')
    group.add_argument(
        "start_time", nargs="?",
        type=lambda s: datetime.strptime(s, '%Y-%m-%dT%H:%M:%S').replace(second=0,
                                                                         microsecond=0,
                                                                         tzinfo=timezone.utc),
        help="Start processing at given time. Use format: %%Y-%%m-%%dT%%H:%%M:%%S "
             "(i.e: 2020-01-01T02:00:00)")
    parser.add_argument(
        "end_time", nargs=1,
        type=lambda s: datetime.strptime(s, '%Y-%m-%dT%H:%M:%S').replace(second=0,
                                                                         microsecond=0,
                                                                         tzinfo=timezone.utc),
        help="Stop processing at given time. Use format: %%Y-%%m-%%dT%%H:%%M:%%S "
             "(i.e: 2020-01-01T02:03:00)")
    parser.add_argument('--writescreen', action='store_true',
                        help="Write results to screen, disabled by default")
    parser.add_argument('--no-savefile', action='store_false',
                        dest='savefile',
                        help="Write results to file, disabled by default")
    parser.add_argument('--saveimage', action='store_true',
                        help="Save image to file, disabled by default")
    parser.add_argument('--force-overwrite', action='store_true',
                        help="Overwrite resultfile instead of appending")
    parser.add_argument('--output-dir',
                        default=default_output_dir,
                        help="Specify output directory (default: %(default)s)")
    parser.add_argument('--output-fname',
                        default=default_output_fname,
                        help="Specify filename format (default: %(default)s)")

    parser.add_argument('--temppath',
                        default=default_temppath,
                        help="Specify temporary directory (default: %(default)s)")

    parser.add_argument('--server',
                        default=default_server,
                        help="Specify server (default: %(default)s)")
    parser.add_argument('--server-dir',
                        default=default_server_dir,
                        help="Specify directory format (default: %(default)s)")
    parser.add_argument('--allsky-fname',
                        default=default_allsky_fname,
                        help="Specify filename format (default: %(default)s)")
    parser.add_argument('--local', action='store_true',
                        help="Use local access to files, disabled by default")

    args = parser.parse_args()

    args.temppath = Path(args.temppath)

    levels = {
        'critical': logging.CRITICAL,
        'error': logging.ERROR,
        'warn': logging.WARNING,
        'warning': logging.WARNING,
        'info': logging.INFO,
        'debug': logging.DEBUG
    }
    level = levels.get(args.log.lower())
    if level is None:
        raise ValueError(
            f"log level given: {args.log}"
            f" -- must be one of: {' | '.join(levels.keys())}")
    logger.setLevel(level=level)

    # When argparse uses the lambda we get a list of datetimes back, use the first
    if type(args.end_time) is list:
        args.end_time = args.end_time[0]
    if type(args.start_time) is list:
        args.start_time = args.start_time[0]

    args.logger = logger
    # args.server = Path(args.server)
    # args.server_dir = Path(args.server_dir)
    # args.allsky_fname = Path(args.allsky_fname)

    logger.debug("delta:           {}".format(args.delta))
    logger.debug("range:           {}".format(args.range))
    logger.debug("parse:           {}".format(args.parse))
    logger.debug("start_time:      {}".format(args.start_time))
    logger.debug("end_time:        {}".format(args.end_time))
    logger.debug("writescreen:     {}".format(args.writescreen))
    logger.debug("savefile:        {}".format(args.savefile))
    logger.debug("saveimage:       {}".format(args.saveimage))
    logger.debug("force-overwrite: {}".format(args.force_overwrite))
    logger.debug("output-dir:      {}".format(args.output_dir))
    logger.debug("output-fname:    {}".format(args.output_fname))
    logger.debug("temppath:        {}".format(args.temppath))
    logger.debug("server:          {}".format(args.server))
    logger.debug("server_dir:      {}".format(args.server_dir))
    logger.debug("allsky-fname:    {}".format(args.allsky_fname))
    main(args)
