# --- python3 program ---
# judge_aurora_ASC.py
#   - v4 by M. Yamauchi 2021-03: filter has changed/output format
#   - v3 by M. Yamauchi 2021-02: debugging and re-structuring.
#   - v2 by M. Yamauchi 2020-08: competely new
#   - v1 by Dennis van Dijk 2016-08: for old ASC (Nikon D700 camera)
#
#  sub-modules to identify auroral pixels in ASC JPEG images
#
## In main python program, iclude as
# import judge_aurora_ASC2021.py as judge_ASC
#           for  D700 lenz (Kiruna: until 2020.4, Abisko: from 2016)
#           or   Sony lenz (Kiruna: from 2020.8)
#
#----+----1----+----2----+----3----+----4----+----5----+----6----+----7- [0:71]
#   note: Email warning criterion of strong aurora
#   LEVEL 6 (touristic strong aurora)
#     L3 >= 8 and  %arc >= 3%   and  %strong >= 0.2% (renew warning every 5-6 minutes)
#----+----1----+----2----+----3----+----4----+----5----+----6----+----7- [0:71]
# Explanation of this module
#--------#---------#---------#---------#---------#---------#---------#-- [1:72]
## Method: fast processing without processed image
## Part 1: Find out URL address of perid t0 - t1
#       fileperiod = list_kirunaASC(t0, t1, 0)
#           --> classify the period as 1, 2, 3 for Nikon camena, Sony test period, Sony nominal mode
#       urllist = list_kirunaASC(t0, t1, *)
#           --> make a list of direct URL for ASC images dring time period t0-t1
#
## Part 2: Convert between jpg-file /numpy-file <--> R,G,B <--> H,L,S
#   H,L,S                    = RGB_everything_HLS(picca)
#   H,L,S, R,G,B, totalpixel = jpg_everything_HLS(jpgname, camera)
#       ## jpgname:  jpg filename of the all-sky image
#       ## picca:   numpy-processed jpg image (after imported using PIL.Image.open)
##    reverse convertion
#   rgb          = RGB_to_fig(R, G, B,  0)     # numpy image
#   im_out       = RGB_to_fig(R, G, B,  *)     # jpg image
#   rgb          = HLS_to_fig(H2,L2,S2, 0)     # numpy image
#   im_out       = HLS_to_fig(H2,L2,S2, *)     # jpg image
#
## Part 3: main calculation of index (Judge each pixel from R,G,B,H,L,S values)
#  *,*,*,,, = derive_mask(H,L,S, R,G,B, camera, writescreen, savefile, saveimage,
#                         moonradius0, filter_level)
#       ## H[*,*,*], L[*,*,*],S[*,*,*]: Hue (color spectrum), Lightness, Saturaton
#       ## camera: so far "Kiruna" or "Abisko"
#       ## count_loop: how many times is this loop is called
#           (if count_loop == 1, Header infomation is printed)
#       ## filter_level = *0**: maskK = full mask within radius
#                       = *1**: maskM = only dense part of maskM_core
#                       = *2**: maskM_full = pre-defined mask without considering moon
#               as strong/weak aurora or cloud/moon/etc and calculate index
#           -->
#           After quick survey if it is worth searchng moon (if aurora and moon coexist),
#           derive the moon location if needed to re-difine the moon filter
#               H (Hue) is the color spectrum.
#               L (Lightness) is degree of darkness (1=brightest, 0=darkest).
#                   L is an easy indicator for the strength of an aurora (in dark sky).
#               S (saturaton) is clearness of the color (1=full color, 0=black-white)
#           --> We define both diffuse & strong aurora
#     note:  Color of aurora in RGB pictures changes with moon and twilight.
#            Therefore, fine tuning is needed for R,G,B,H,L,S ranges.
#
## Part 4: calculate index using final mask and Luminocity (L) value
#  cntMaybe,cntYes,cntStrong, AverageLumS, expL_strong, LexpL_strong, LLL_strong, L_lowest,L_highest
#           = judge_ASC.HLS_to_index(L,R,G,maskW,maskA,maskS,camera,count_loop)
#
#     --> return(Hstrong,Lstrong,Sstrong,cntMaybe, cntYes, cntStrong, cntCloud,
#                  cntMoon, AverageLumS, expL_strong, LexpL_strong, LLL_strong)
#
## Part 5: image processing (use reverse conversion of Part 2)
#   Return
#     im_out:  equivalent to the imported jpg image using PIL.Image.open
#       or
#     picca:   numpy-array of RGB image, float64
#
#--------#---------#---------#---------#---------#---------#---------#
## note for H(Hue), L(Lightness), and S(Saturation)
#         H (Hue) is the color spectrum.
#         L (Lightness) is degree of darkness (1=brightest, 0=darkest).
#             L is an easy indicator for the strength of an aurora (in dark sky).
#         S (saturaton) is clearness of the color (1=full color, 0=black-white)
#     --> We define aurora (bot diffuse+strong), strong aurora, cloud, moon
#  note:  Color of aurora in RGB pictures changes with moon and twilight.
#      Therefore, fine tuning is needed for H, S, L ranges.
#--------#---------#---------#---------#---------#---------#---------#

from datetime import datetime, timedelta
#import urllib as web          # for python 2
import urllib.request as web   # for python 3, Use as web.urlopen() and web.urlretrieve()
import urllib.error as weberror
import numpy as np
import matplotlib.image as mpimg    # Use only as mpimg.imread()
#import matplotlib.pyplot as plt
import colorsys
import math
#import glob   # to read files in a folder (2020-08, Futaana), use only as glob.glob()
import re
from PIL import Image   # Use only as Image.open()

DATE_TIME_STRING_FORMAT = '%Y-%m-%dT%H:%M'

NEW_ALLSKY_DATE_TIME_STRING_FORMAT = 'https://www.irf.se/alis/allsky/krn/%Y/%m/%d/%H/'
NEW_ALLSKY_JPEG_FORMAT = NEW_ALLSKY_DATE_TIME_STRING_FORMAT + '%Y-%m-%dT%H.%M.%S.000KRN_small.jpeg'
OLD_ALLSKY_DATE_TIME_STRING_FORMAT = 'https://www2.irf.se/allsky/%Y/%Y%m%d/jpgs/'

# .strftime converts time to string, like '2020-04-24T12:00'
#   kirunaASC_epoch1: file format changes from old format to new format
#   kirunaASC_epoch2: camera setting (exposure etc) is finally fixed
#
kirunaASC_epoch1 = datetime.strptime('2020-04-24T12:00', DATE_TIME_STRING_FORMAT)
#kirunaASC_epoch1b = datetime.strptime('2020-10-20T12:00', DATE_TIME_STRING_FORMAT)
kirunaASC_epoch2 = datetime.strptime('2021-01-23T12:00', DATE_TIME_STRING_FORMAT)


#========#=========#=========#=========#=========#=========#========-#
# Part 1: Find out URL address of perid t0 - t1
#       fileperiod = list_kirunaASC(t0, t1, 0)
#           --> classify the period as 1, 2, 3 for Nikon camena, Sony test period, Sony nominal mode
#       urllist = list_kirunaASC(t0, t1, *)
#           --> make a list of direct URL for ASC images dring time period t0-t1
#
# Note: submodue that is called in list_kirunaASC(t0, t1, *)
#          as html = list_old_kirunaASC(t0, t1) ==> list all URL over a night duing "while loop"
#       or as html = list_new_kirunaASC(t0, t1) ==> list URL for each hour duing "while loop"
#  - Returns a list of all URL of all-sky-camera images on specific night/hour
#  - File error cases are given as except
#--------#---------#---------#---------#---------#---------#---------#
def list_old_kirunaASC(t0, t1):
    filelist_day, urllist = [], []
    if t0.hour < 8: t0 = (t0 - timedelta(days=1)).replace(hour=14)
    elif t0.hour < 14: t0 = t0.replace(hour=14)
    if t1.hour >= 9 and t1.hour <= 14: t1 = t1.replace(hour=8)
    if t0 >= t1: t1 = t0 + timedelta(minutes=1)
    while t0 < t1:
        url = t0.strftime(OLD_ALLSKY_DATE_TIME_STRING_FORMAT)
        # strftime comverts time to string, like 'https://www2.irf.se/allsky/2017/20170907/jpgs'
        try:
            website_day = web.urlopen(url)
            filelist_day = [line.decode('utf-8') for line in website_day]
        except weberror.HTTPError as e:
            # print('The server couldn\'t fulfill the request.')
            # print('Error code: ', e.code)
            pass
        except weberror.URLError as e:
            # print('We failed to reach a server.')
            # print('Reason: ', e.reason)
            pass
        else:
            # everything is fine        except weberror.URLError as e:
            pass
        
        ## regex & [regex....] extracts only normal file names in the folder
        #  (like KRN20170908T014700E01000Q.JPG)
        regex = re.compile('href="KRN20[\s\S]*.JPG"')
        filelist_day = [regex.search(line).group(0).split('"')[1]
                        for line in filelist_day
                        if regex.search(line) is not None]
        filelist_day = np.sort(filelist_day, None)
        for jpgname in filelist_day:
            urlfname = "".join((url,jpgname))
            urllist.append(urlfname)

        t0 += timedelta(hours=24)
    return urllist


#--------#---------#---------#---------#---------#---------#---------#
def list_new_kirunaASC(t0, t1):
    filelist_day, urllist = [], []
    if t1.minute==0 and t1 > t0: t1 = t1 - timedelta(minutes=1)
    elif t1.minute < t0.minute:  t1 = t1.replace(minute=59)
    while t0 <= t1:
        url = t0.strftime(NEW_ALLSKY_DATE_TIME_STRING_FORMAT)
        # strftime converts time to string, like 'https://www.irf.se/alis/allsky/krn2020/11/02/23/'
        try:
            website_day = web.urlopen(url)
            filelist_day = [line.decode('utf-8') for line in website_day]
        except weberror.HTTPError as e:
            # print('The server couldn\'t fulfill the request.')
            # print('Error code: ', e.code)
            pass
        except weberror.URLError as e:
            # print('We failed to reach a server.')
            # print('Reason: ', e.reason)
            pass
        else:
            # everything is fine        except weberror.URLError as e:
            pass
        
        ## regex & [regex....] extracts only normal file names in the folder
        #  (like KRN20170908T014700E01000Q.JPG)
        regex = re.compile('href="20[\s\S]*_small.jpeg"')
        filelist_day = [regex.search(line).group(0).split('"')[1]
                        for line in filelist_day
                        if regex.search(line) is not None]
        filelist_day = np.sort(filelist_day, None)
        for jpgname in filelist_day:
            urlfname = "".join((url,jpgname))
            urllist.append(urlfname)
                
        t0 += timedelta(hours=1)

    return urllist


#--------#---------#---------#---------#---------#---------#---------#
# read only specified time (otherwise, latest file)
#--------#---------#---------#---------#---------#---------#---------#
def nowcast_url_kirunaASC(t1):
    NOWCAST_STRING_FORMAT = 'https://www.irf.se/alis/allsky/krn/%Y/%m/%d/%H/%Y-%m-%dT%H.%M.00.000KRN_small.jpeg'
    urlfname  = t1.strftime(NOWCAST_STRING_FORMAT)
    # strftime converts time to string, like https://www.irf.se/alis/allsky/krn/2021/03/13/20/2021-03-13T20.47.00.000KRN_small.jpeg
    print("target filename: ", urlfname)

    flag = 0
    try:
        #image  = Image.open(urlfname)  ## <== this does not work
        (image, msg) = web.urlretrieve(urlfname)
    except IOError:    #This means that the file does not exist (or some other IOError)
        print ("No file by that name")
        flag = 1
        urlfname = 'https://www.irf.se/alis/allsky/krn/latest_small.jpeg'
        try:
            #urlfname = 'https://www.irf.se/alis/allsky/krn/latest_small.jpeg'
            (image, msg) = web.urlretrieve(urlfname)
        except IOError:    # This means that the file does not exist (or some other IOError)
            print("No file by that name1")
            flag = 2
            #pass
        else:
            # everything is fine        except weberror.URLError as e:
            pass
    else:
        # everything is fine        except weberror.URLError as e:
        pass


    print("filename, error_code (in submodule): ", urlfname, flag)

    return urlfname, flag


#--------#---------#---------#---------#---------#---------#---------#
# returns file-list for period [from_date_time, to_date_time]
#    until 2020.04 = list_old_kirunaASC(t0, t1) ==> list all jpg file over night duing "while loop"
#    from  2020.08 = list_new_kirunaASC(t0, t1) ==> list jpg files for each hour duing "while loop"
# t0-t1: period to examine,  flag = 0: define only fileperiod, >= 1: obtain file
def list_kirunaASC(t0, t1, flag):
    if   t1 < kirunaASC_epoch1: fileperiod = 1
    elif t1 < kirunaASC_epoch2: fileperiod = 2
    else: fileperiod = 3
    
    if flag == 0: return fileperiod
    
    #- else -#---------#---------#---------#---------#---------#

    if t1 < kirunaASC_epoch1: urllist = list_old_kirunaASC(t0, t1)
    else:                     urllist = list_new_kirunaASC(t0, t1)
    return urllist


#========#=========#=========#=========#=========#=========#========-#
# Part 2: Convert between jpg-file /numpy-file <--> R,G,B <--> H,L,S
#   H,L,S                    = RGB_everything_HLS(picca)
#   H,L,S, R,G,B, totalpixel = jpg_everything_HLS(jpgname, camera)
#   rgb                      = RGB_to_fig(R, G, B,  0)     # numpy image
#   im_out                   = RGB_to_fig(R, G, B,  *)     # jpg image
#   rgb                      = HLS_to_fig(H2,L2,S2, 0)     # numpy image
#   im_out                   = HLS_to_fig(H2,L2,S2, *)     # jpg image
#--------#---------#---------#---------#---------#---------#---------#
def RGB_everything_HLS(picca):
    #  --- python3 program ---
    #  @rgb to hls author: Paul, edited by: Dennis and SailAvid
    # The transformation codes from Wikipedia are used to transform.
    #
    # input: picca = numpy-processed ASC jpg
    #     --> Convert each pixel from RGB to HLS
    # otput: H,L,S = RGB_everything_HLS(AllSkyPicture)
    #       H[*,*,*]: Hue = the color spectrum.
    #       L[*,*,*]: Lightness = degree of darkness (1=brightest, 0=darkest).
    #           L is an easy indicator for the strength of an aurora (in dark sky).
    #       S[*,*,*]: Saturaton = clearness of the color (1=full color, 0=black-white)
    
    R2, G2, B2                      = picca[:,:,0], picca[:,:,1], picca[:,:,2]
    M                               = np.amax(picca, axis=2)  # maximum of R, G, B values
    m                               = np.amin(picca, axis=2)  # minimum of R, G, B values
    C                               = M-m #chroma
    Cmsk                            = C!= 0
    #Hue:
    H                               = np.zeros(R2.shape, float)
    mask                            = (M==R2)&Cmsk
    H[mask]                         = np.mod((G2[mask]-B2[mask])/C[mask], 6)
    mask                            = (M==G2)&Cmsk
    H[mask]                         = ((B2[mask]-R2[mask])/C[mask] + 2)
    mask                            = (M==B2)&Cmsk
    H[mask]                         = ((R2[mask]-G2[mask])/C[mask] + 4)
    H                               /= 6
    #Lightness:
    L                               = 0.5*(M+m)
    #Saturation:
    S                               = np.zeros(R2.shape, float)
    S[Cmsk]                         = ((C[Cmsk])/(1-np.absolute(2*L[Cmsk]-1)))
    S[S>1.0]                        = 1
    
    return(H,L,S)


def jpg_everything_HLS(jpgname, camera):
    #  --- python3 program ---
    # v 20210303
    #   Add "reading" part before RGB_everything_HLS(picca):
    #   = define the pixels (circle area) to analyse, making non-relevant are black (R=G=B=0)
    #     -> total pixels are also connted
    #
    ## picca.shape[X,Y,Z] => 90 degree clockwize converted X-Y-Z Cartecian
    #       X = vertical downward (north = 0, south = max)
    #       Y = holizontal rightward (east = 0, west = max)
    #       Z = [0,1,2] is RGB values
    #
    
    img255  = Image.open(jpgname)
    if img255.mode != "RGB":
        img255 = img255.convert("RGB")

    if camera == 'Kiruna' or camera == 'Kiruna0' :  # up to 2020.04, 455x455
        #picca = (np.array(img255)/255.0)[14:469,130:585,:]   # 455x455 = minimum size
        picca = (np.array(img255)/255.0)[9:474,125:590,:]     # 465x465 = better for moon detection
    elif camera == 'Kiruna2': # from 2020.08, 446x446
        picca = np.array(img255)/255.0
        picca = picca[4:450,5:451,:]   # 446x446 = minimum size
        picca = picca[8:438,8:438,:]   # cut a part of edge 430X430
    elif camera == 'Kiruna3': # from 2020.10.28, 466x466 (extra black 10 pixels at eadge)
        picca = np.array(img255)/255.0
        picca = picca[14:460,15:461,:] # first cut to the minimum size
        picca = picca[8:438,8:438,:]   # cut more part of edge 430X430


    xdim,ydim,zdim = picca.shape # 430x430 or 465x465
    radx,rady, distance2 = xdim/2,ydim/2, (xdim+1)*(ydim+1)/4
    if camera == 'Kiruna3': distance2 = distance2 - 5
    for x in range(0,xdim):
        for y in range(0,ydim):
            if (x-radx)*(x-radx) + (y-rady)*(y-rady) > distance2: picca[x,y,:] = 0

    # New from version 2021-2-26
    #   After moon filter is defined, we trim upper left (west-north-west) to skip "city light part" from analyses
    if camera == 'Kiruna3': # SONY
        for x in range(0,xdim):
            for y in range(0,ydim):
                if (-0.4*x) + y > 320: picca[x,y,:] = 0     # best is >300 but that trip is after moon detection

    R,G,B = picca[:,:,0], picca[:,:,1], picca[:,:,2]
    totalpixel = np.count_nonzero(R+G+B)
    H,L,S = RGB_everything_HLS(picca)
    return(H,L,S,  R,G,B, totalpixel)


#------------------------------------
#prepare SECTION (5) image file for only strong case
def Stitch_Plot(image1,image2,image3,image4):
    totalImg                    = np.hstack((image1,image2,image3,image4))
    return(totalImg)
    # Stitch_Plot( , , , ) function sticks the images together before saving.
    # The first image is the original one,
    # while the other 3 are the three different filters.
        

#------------------------------------
# image_out = RGB_to_fig(R,G,B, 0)
# jpg_out   = RGB_to_fig(R,G,B, 1)
def RGB_to_fig(R,G,B, output_type):
    #  --- python3 program ---
    # input:  R[*,*], G[*,*], B[*,*]
    #         output_type = 0: image file
    #                     = 1: jpg file
    # output: picca = All Sky RGB[*,*,*] Picture
    
    rgb = np.zeros((R.shape[0],R.shape[1],3), 'float')
    rgb[..., 0], rgb[..., 1], rgb[..., 2] = R, G, B

    # rgb [*,*,*] is numpy array  but not jpg image.
    if output_type == 0: return(rgb)
    
    #rgb8 = np.int_(rgb*255)
    rgb8 = np.uint8(rgb*255)
    im_out = Image.fromarray(rgb8, mode='RGB')
    
    # now is jpg image.
    return(im_out)


#------------------------------------
#prepare SECTION (4) This part transforms back from HLS to RGB
# image_out = HLS_to_fig(H2,L2,S2,0)
# jpg_out   = HLS_to_fig(H2,L2,S2,1)
def HLS_to_fig(H2,L2,S2,output_type):
    #  --- python3 program ---
    #  @hls to rgb author: SailAvid, edited by: Dennis
    # The transformation codes from Wikipedia are used to transform.
    #
    # input:  H[*,*], L[*,*], S[*,*]
    #         output_type = 0: image file
    #                     = 1: jpg file
    #     --> Convert each pixel from HLS to RGB[*,*,*]
    #      &  re-construct a picture
    # output: picca = All Sky RGB Picture
    
    C                           = (1-np.absolute(2*L2-1))*S2
    H2                          *= 6
    X                           = C*(1-np.absolute(np.mod(H2,2)-1))
    R                           = np.zeros(H2.shape, float)
    G                           = np.zeros(H2.shape, float)
    B                           = np.zeros(H2.shape, float)
    mask = (H2>=0)              == (H2 < 1)
    R[mask],G[mask]             = C[mask], X[mask]
    mask = (H2>=1)              == (H2 < 2)
    R[mask],G[mask]             = X[mask], C[mask]
    mask = (H2>=2)              == (H2 < 3)
    G[mask],B[mask]             = C[mask], X[mask]
    mask = (H2>=3)              == (H2 < 4)
    G[mask],B[mask]             = X[mask], C[mask]
    mask = (H2>=4)              == (H2 < 5)
    R[mask],B[mask]             = X[mask], C[mask]
    mask = (H2>=5)              == (H2 < 6)
    R[mask],B[mask]             = C[mask], X[mask]
    m                           = L2 - 0.5 * C
    R, G, B                     = R+m, G+m, B+m
    
    rgb = np.zeros((R.shape[0],R.shape[1],3), 'float')
    rgb[..., 0], rgb[..., 1], rgb[..., 2] = R, G, B

    # rgb [*,*,*] is numpy array  but not jpg image.
    if output_type == 0: return(rgb)
    
    #rgb8 = np.int_(rgb*255)
    rgb8 = np.uint8(rgb*255)
    im_out = Image.fromarray(rgb8, mode='RGB')
    
    # now is jpg image.
    return(im_out)


#========#=========#=========#=========#=========#=========#========-#
# Part 3: main calculation of index (Judge each pixel from R,G,B,H,L,S values)
#               as strong/weak aurora or cloud/moon/etc and calculate index
#           -->
#           After quick survey if it is worth searchng moon (if aurora and moon coexist),
#           derive the moon location if needed to re-difine the moon filter
#               H (Hue) is the color spectrum.
#               L (Lightness) is degree of darkness (1=brightest, 0=darkest).
#                   L is an easy indicator for the strength of an aurora (in dark sky).
#               S (saturaton) is clearness of the color (1=full color, 0=black-white)
#           --> We define both diffuse & strong aurora
#     note:  Color of aurora in RGB pictures changes with moon and twilight.
#            Therefore, fine tuning is needed for R,G,B,H,L,S ranges.
#--------#---------#---------#---------#---------#---------#---------#
##  *,*,*,,, = derive_mask(H,L,S, R,G,B, camera, writescreen, savefile, saveimage,
#                          moonradius0, filter_level)
#       ## filter_level = *0**: maskK = full mask within radius
#                       = *1**: maskM = only dense part of maskM_core
#                       = *2**: maskM_full = pre-defined mask without considering moon
#
## sub-modules:
#    maskW,maskA,maskS, cntMaybe,cntYes,cntStrong = aurorafilter(H,L,S, R,G,B, camera, filter_level)
#       ## filter_level = ***0: HLS for both cameras (Nikon & SONY)
#                       = ***1: RGB for SONY camera
#    maskL, cntLight = lightfilter(H,L,S, R,G,B, camera)
#       ## filter_level = ***0: core part only
#                       = ***1: extra part added (for survey)
#
#    maskC, cntCloud_raw = cloudfilter(H,L,S, R,G,B, camera, filter_level)
#
#    maskM_core, maskM_blight, maskM_full, ,,,,
#       = moonfilter(H,L,S, R,G,B, camera, 0)
#       ## filter_level = **0* - **4*: moon filter (core part + without/with surrounding part) <== default
#                       = **5* - **9*: + filling boundaries
#    M,M2, , ,,,, = mooncenter(L, maskM_core, maskM_blight, maskM_full,camera, radius, filter_level)
#       ## filter_level = **0* or **5*: use maskM_core  <== to find out center
#                       = **1* or **6*: use maskM_blight
#                       = **2* or **7*: use maskM_full <== when filling 14*raidus distance from moon
#    cntMaybe,cntYes,cntStrong, ,,,, = HLS_to_index(L,R,G,maskW,maskA,maskS,camera,totalpixel,count_loop)
#
## sub-sub-modules:
#       HLS_max_min(camera)
#       RGB_mooncriterion(camera)
#       cloudcriterion(camera)
#--------#---------#---------#---------#---------#---------#---------#
def derive_mask(H,L,S, R,G,B, camera, totalpixel, writescreen, savefile, saveimage, moonradius0, filter_level, count_loop):
    # maskW,maskA,maskS, cntMaybe,cntYes,cntStrong,cntVoid,cntDark,
    #   AverageLumS, expL_strong, LexpL_strong, LLL_strong, radiusL,new_radius,flag,
    #   cntLight,cntCloud_raw, cntMoon0, Xcenter0, Ycenter0,
    #   cntInside, Xcenter_local, Ycenter_local, cntMoon2, Xcenter2, Ycenter2
    #   = derive_mask(H,L,S, R,G,B, camera, totalpixel, writescreen, savefile, saveimage,
    #     moonradius0, filter_level, count_loop)
    # input:    H[:,:],L,S, R,G,B = HLS-values and RGB values of each pixel
    #           writescreen, savefile, saveimage => 0 : yes
    #                                            <= -1: no
    #           moonradius0 = initial value (normally = 5)
     #-------------------------
    
    # (1) quick survey if it is worth searching moon
    maskW,maskA,maskS, cntMaybe,cntYes,cntStrong = aurorafilter(H,L,S, R,G,B, camera, 1) # ***0=HLS, ***1=RGB
    maskM_core, maskM_blight, maskM_full, cntMoon_core,cntMoon_blight,cntMoon_full = moonfilter(H,L,S, R,G,B, camera, 0) # **5* - **9* = filling
    maskL, cntLight = lightfilter(H,L,S, R,G,B, camera)
    maskC_core, cntCloud_core = cloudfilter(H,L,S, R,G,B, camera, 0)   # parameter 0: core part only
    maskC_full, cntCloud_full = cloudfilter(H,L,S, R,G,B, camera, 1)   # parameter 1: all possible cloud

    maskMLC_survey = (maskM_blight == 1) | (maskL == 1) | (maskC_full == 1) # pixel to be void = True =1
    maskW = (maskW == 1) & (maskMLC_survey == 0)     # we count weak light as cloud rather than diffuse aurora
    maskA = (maskA == 1) & (maskMLC_survey == 0)
    maskS = (maskS == 1) & (maskMLC_survey == 0)
    cntMaybe,cntYes,cntStrong = np.count_nonzero(maskW),np.count_nonzero(maskA),np.count_nonzero(maskS)

    maskMLC = (maskM_full == 1) | (maskL == 1) | (maskC_full == 1)  # pixels outside the camera = black, which means maskMLC=0
    cntVoid = np.count_nonzero(maskMLC)             # void area, which has G>0 and hence "totalpixel" is actual total pixel
    
    if writescreen:
        print(' survey (Maybe,Yes,Strong)={0:d}, {1:d}, {2:d} void{3:d}%(Cloud,Moon,light={4:d}, {5:d}, {6:d})  '.format(cntMaybe,cntYes,cntStrong, int(cntVoid), cntCloud_full, cntMoon_full, cntLight))

    shortcut = 0    # >=1: skip searching moon (either no aurora or cloudy)
    if camera == 'Kiruna3':
        if cntVoid > 0.85*totalpixel:
            if cntMaybe < 5000:                     shortcut = 1
            elif cntMaybe < 2000 and cntYes < 50:   shortcut = 2
        elif cntVoid > 0.55*totalpixel:
            if cntMaybe < 3000:                     shortcut = 3
            elif cntMaybe < 1000 and cntYes < 30:   shortcut = 4
        elif cntMaybe < 1500:                       shortcut = 8
        elif cntMaybe < 500 and cntYes < 15:        shortcut = 9
    elif camera == 'Kiruna' or camera == 'Kiruna0' or camera == 'Abisko':
        if cntVoid > 0.85*totalpixel:
            if cntMaybe < 3000:                     shortcut = 11
            elif cntYes < 200:                      shortcut = 12
            elif cntMaybe < 2000 and cntYes < 100:  shortcut = 13
        elif cntVoid > 0.60*totalpixel:
            if cntMaybe < 2000:                     shortcut = 14
            elif cntYes < 100:                      shortcut = 15
            elif cntMaybe < 1000 and cntYes < 50:   shortcut = 16
        elif cntMaybe < 1000:                       shortcut = 18
        elif cntMaybe < 500 and cntYes < 20:        shortcut = 19
    else:
        if cntVoid > 0.80*totalpixel:
            if cntMaybe < 10000:                    shortcut = 21
            elif cntYes < 500:                      shortcut = 22
            elif cntMaybe < 5000 and cntYes < 200:  shortcut = 23
        elif cntVoid > 0.50*totalpixel:
            if cntMaybe < 5000:                     shortcut = 25
            elif cntYes < 200:                      shortcut = 26
            elif cntMaybe < 2000 and cntYes < 100:  shortcut = 27
        elif cntMaybe < 2000:                       shortcut = 57
        elif cntMaybe < 1000 and cntYes < 200:      shortcut = 58

    # (2) define final maskW, maskA, maskS
    # (2a) skip detecting moon because of no aurora
    if shortcut >= 1 or int(filter_level/100) % 10 >= 2:
        maskW = (maskW == 1) & (maskMLC == 0)     # we count weak light as cloud rather than diffuse aurora
        maskA = (maskA == 1) & (maskMLC == 0)
        maskS = (maskS == 1) & (maskMLC == 0)
        cntMoon0, Xcenter0, Ycenter0, cntInside, Xcenter_local, Ycenter_local, cntMoon2, Xcenter2, Ycenter2, radiusL,new_radius,flag = cntMoon_full,-2,-2, cntMoon_blight,-2,-2, cntMoon_core,-2,-2, -2,-2,999
        cntCloud0 = cntCloud_full
    
    # (2b) full examination of the moon
    else:
        M,M2,cntMoon0,cntInside,cntMoon2,Xcenter0,Ycenter0,Xcenter_local,Ycenter_local,Xcenter2,Ycenter2, radiusL,new_radius,flag = mooncenter(L, maskM_core, maskM_blight, maskM_full, camera, moonradius0, filter_level)
        # input:  filter_level = input mask scheme (0=maskM_core, 10=maskM_blight, 20=maskM_full)
        # output: M(*,*)=most restrictive, M2(*,*)=filter also surrounding with 5*radius and 14*radius
        #               1 = True  = moon-affected pixels (to be masked)
        #               0 = False = clear sky
        #         local method (*_local) and high-intensity local method (*2)
        #         cntMoon: numbers of moon-pixels, identified by
        #         Xcenter,Ycenter: center location in pixel (numpy array), by
        #              mask(***0), M(no mark), and small subregion (***_local)
        if   int(filter_level/100) % 10 == 0: maskM = M2   # full mask within radius
        elif int(filter_level/100) % 10 == 1: maskM = M    # only dense part of maskM_core
        else: maskM = M2

        maskML = (maskM == 1) | (maskL == 1)
    
        maskW = (maskW == 1) & (maskML == 0)    # we count weak light as aurora rather than cloud
        maskA = (maskA == 1) & (maskML == 0)
        maskS = (maskS == 1) & (maskML == 0)

        maskMLC = (maskC_core == 1) | (maskML == 1) # 1 = dark sky, 0 = masked.  Use limited mask for cloud when aurora exists
        cntCloud0 = cntCloud_core

        if writescreen: print('flag={:d}, Moon: {:d}({:d},{:d}) > {:d}({:d},{:d}) > {:d}({:d},{:d})'.format(flag, cntMoon0,Xcenter0,Ycenter0, cntMoon2,Xcenter2,Ycenter2, cntInside,Xcenter_local,Ycenter_local))
                
    # New from version 2021-2-26 & 3-03
    #   After moon filter is defined, we trim upper left (west-north-west) to skip "city light part" from analyses
    xdim,ydim = G.shape # 430x430 or 465x465
    if camera == 'Kiruna3': # SONY
        for x in range(0,xdim):
            for y in range(0,ydim):
                if (-0.4*x) + y > 300: G[x,y],L[x,y] = 0,0
        totalpixel = np.count_nonzero(R+G+B)   # reduced total area

    cntVoid = np.count_nonzero( maskMLC * R )       # void area, which has G>0 and hence "totalpixel" is actual total pixel
    cntDark = np.count_nonzero( (1-maskMLC) * R )   # pixels that are not maskd within the active (non-black) area
    cntVoid, cntDark = cntVoid*100./totalpixel, cntDark*100./totalpixel

    # (3) define index from maskW, maskA, maskS
    cntMaybe,cntYes,cntStrong, AverageLumS, expL_strong, LexpL_strong, LLL_strong, L_lowest,L_highest = HLS_to_index(L,R,G,maskW,maskA,maskS,camera,totalpixel,count_loop)

    if writescreen:
        print (' flag={0:d}, Maybe,Yes,Strong={1:d}, {2:d}, {3:d} void{4:d}%(Cloud,Moon,light={5:d},{6:d},{7:d}), <L>, L**3={8:.2f},{9:.3f}, {10:.2f}<L<{11:.2f} '.format(flag,cntMaybe, cntYes, cntStrong, int(cntVoid), cntCloud0, max(cntMoon0, cntMoon2), cntLight, AverageLumS, LLL_strong, L_lowest,L_highest))
        #print(' flag=',flag, ' cloud+moon+light+dark = total pixels', cntCloud0, cntMoon, cntLight, cntDark, cntMoon+cntLight+cntCloud_full+cntDark)

    return(maskW,maskA,maskS, cntMaybe,cntYes,cntStrong,cntVoid,cntDark, AverageLumS, expL_strong, LexpL_strong, LLL_strong, radiusL,new_radius,flag, cntLight,cntCloud0, cntMoon0, Xcenter0, Ycenter0, cntInside, Xcenter_local, Ycenter_local, cntMoon2, Xcenter2, Ycenter2)
    

#========#=========#=========#=========#=========#=========#========-#
# Part 3b sub-modules:
#
#--------#---------#---------#---------#---------#---------#---------#
def HLS_max_min(camera):
    # Define filter to judge each pixel as 'diffuse + arc' aurora, aurura arc, strong aurora, & cloud
    # --- filtering version 2020-08 for Nikon, 2020-10 for SONY)
    ##old 2019 version (Hmin, Hmax, Smin, Smax, Lmin, Lmax = 0.2, 0.46, 0.09, 0.8, 0.05, 0.8)   #  Weak aurora
    Hmin_w, Hmax_w, Smin_w, Smax_w, Lmin_w, Lmax_w = 0.16, 0.50, 0.09, 0.8,  0.05, 0.80   #  Weak aurora
    Hmin_a, Hmax_a, Smin_a, Smax_a, Lmin_a, Lmax_a = 0.18, 0.46, 0.13, 0.8,  0.15, 0.80   #  Aurora
    Hmin_s, Hmax_s, Smin_s, Smax_s, Lmin_s, Lmax_s = 0.20, 0.46, 0.17, 0.8,  0.30, 0.80   #  Strong aurora
    Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = 0.00, 0.16, 0.10, 0.8,  0.15, 0.83   #  Cloud
    # H = 90deg(0.25): yellow-green (some cloud), =180-240deg: blue (sky), <0.57: for moon+clear sky, >0.18 for aurora over cloud
    # S = 20%: color is well defined, =9%: faint color, <80%: remove Lider line
    # L = 5%: boundary to darkness,  <83%: romove moon
    if camera == 'Abisko':
        Hmin_w, Hmax_w, Smin_w, Smax_w, Lmin_w, Lmax_w = 0.16, 0.50, 0.00, 0.8, 0.05, 0.90   #  Weak aurora
        Hmin_a, Hmax_a, Smin_a, Smax_a, Lmin_a, Lmax_a = 0.20, 0.46, 0.00, 0.8, 0.10, 0.90   #  Aurora
        Hmin_s, Hmax_s, Smin_s, Smax_s, Lmin_s, Lmax_s = 0.20, 0.46, 0.20, 0.8, 0.20, 0.90   #  Strong aurora
        Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = 0.00, 0.16, 0.00, 1.0, 0.15, 0.83   #  Cloud

    elif camera == 'Kiruna2' or camera == 'Kiruna3': # never used
        Hmin_w, Hmax_w, Smin_w, Smax_w, Lmin_w, Lmax_w = 0.21, 0.44, 0.15, 1.01, 0.29, 0.97   #  Weak aurora
        Hmin_a, Hmax_a, Smin_a, Smax_a, Lmin_a, Lmax_a = 0.24, 0.37, 0.26, 1.01, 0.56, 0.97   #  Aurora
        Hmin_s, Hmax_s, Smin_s, Smax_s, Lmin_s, Lmax_s = 0.24, 0.36, 0.30, 1.01, 0.75, 0.97   #  Strong aurora
        Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = 0.47, 0.21, 0.00, 0.30, 0.20, 0.80   #  Cloud

    return(Hmin_w, Hmax_w, Smin_w, Smax_w, Lmin_w, Lmax_w, Hmin_a, Hmax_a, Smin_a, Smax_a, Lmin_a, Lmax_a, Hmin_s, Hmax_s, Smin_s, Smax_s, Lmin_s, Lmax_s, Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c)


#------------------------------------
def RGB_mooncriterion(camera):
    # Define filter to judge each pixel as moon or house light
    # --- filtering version 2020-10
    # (1) for SONY camera
    Gmin_m,  Gmax_m,  Rratio_m,  Bratio_m  = 0.94, 1.01, 0.99, 1.98  # G>240 (>0.94), (R/G>0.99 or (R+B)/G>1.98)
    Gmin_m2, Gmax_m2, Rratio_m2, Bratio_m2 = 0.64, 1.01, 0.98, 0.96  # G>185 (>0.64), R/G ratio>0.98, B/G>0.96
    Gmin_m3, Gmax_m3, Rratio_m3, Rratio_m4 = 0.40, 0.80, 0.98, 1.05  # G>101 (>0.40), R/G ratio=0.98-1.05
    Gmin_l1, Gmax_l1, Rratio_l1, Bratio_l1 = 0.68, 0.95, 0.83, 0.71  # G=175-242 (0.68-0.95), R/G>0.83, B/G>0.71
    Gmin_l2, Gmax_l2, Rratio_l2, Bratio_l2 = 0.62, 0.69, 0.73, 0.65  # G=163-175 (0.63-0.69), R/G>0.73, B/G < 0.65
    if camera == 'Kiruna2' or  camera == 'Kiruna3':
        return(Gmin_m,Gmax_m,Rratio_m,Bratio_m, Gmin_m2,Gmax_m2,Rratio_m2,Bratio_m2,Gmin_m3, Gmax_m3, Rratio_m3, Rratio_m4,  Gmin_l1,Gmax_l1,Rratio_l1,Bratio_l1, Gmin_l2,Gmax_l2,Rratio_l2,Bratio_l2)
        #Gmin_m,Gmax_m,RGratio_m,BGratio_m,  Gmin_m2,Gmax_m2,Rratio_m2,Bratio_m2, Gmin_m3, Gmax_m3, Rratio_m3, Rratio_m4, Gmin_l1,Gmax_l1,RGratio_l1,BGratio_l1, Gmin_l2,Gmax_l2,RGratio_l2,BGratio_l2 = RGB_mooncriterion(camera)

    else:
    # (2) for Nikon camera
        Gmin_m, Gmax_m, Rratio_m, Bratio_m = 0.92, 1.01, 0.99, 0.97  # G>200 (>0.78), R/G ratio>0.93, B/G>0.86
        Gmin_m2, Gmax_m2, Rratio_m2, Bratio_m2 = 0.64, 0.92, 0.97, 0.95  # G=170-200 (0.66-0.78), R/G>0.97, B/G>0.97

        return(Gmin_m,Gmax_m,Rratio_m,Bratio_m, Gmin_m2,Gmax_m2,Rratio_m2,Bratio_m2)
        #Gmin_m1,Gmax_m1,Rratio_m1,Bratio_m1,  Gmin_m2,Gmax_m2,Rratio_m2,Bratio_m2 = RGB_mooncriterion(camera)


#------------------------------------
# This will NOT be used
def HLS_mooncriterion(camera):
    # Define filter to judge each pixel as moon
    # --- filtering version 2020-08
    Hmin_m, Hmax_m, Smin_m, Smax_m, Lmin_m, Lmax_m = 0.00, 1.01, 0.00, 0.20, 0.88, 1.01    #  Moon
    Hmin_m2, Hmax_m2, Smin_m2, Smax_m2, Lmin_m2, Lmax_m2 = 0.00, 1.01, 0.00, 0.13, 0.75, 1.01    #  Moon
    # maskMoon   = (S<0.2) & (L>0.88)  or  (S<0.13) & (L>0.75)  # moon filter for Kiruna camera at IRF
    #old maskM   = (L>0.80) & (S<0.13)  # moon filter for Kiruna camera at IRF
    #oldold maskM   = (S<0.2)  & (L>0.7)   # moon filter for Kiruna camera at IRF
    if camera == 'Abisko':
        Lmin_m, Lmin_m2 = 0.90, 0.80   #  Moon
    elif camera == 'Kiruna2' or  camera == 'Kiruna3':
        Hmin_m, Hmax_m, Smin_m, Smax_m, Lmin_m, Lmax_m = 0.12, 0.19, 0.40, 1.01, 0.97, 1.00 # Moon
        Hmin_m2, Hmax_m2, Smin_m2, Smax_m2, Lmin_m2, Lmax_m2 = 0.06, 0.21, 0.10, 0.60, 0.93, 0.99 # Moon
    
    return(Hmin_m,Hmax_m, Smin_m,Smax_m, Lmin_m,Lmax_m,  Hmin_m2,Hmax_m2, Smin_m2,Smax_m2, Lmin_m2,Lmax_m2)
    #Hmin_m1, Hmax_m1, Smin_m1, Smax_m1, Lmin_m1, Lmax_m1, Hmin_m2, Hmax_m2, Smin_m2, Smax_m2, Lmin_m2, Lmax_m2 = HLS_mooncriterion(camera)


#------------------------------------
def cloudcriterion(camera):
    # Define filter to judge each pixel as moon or house light
    # --- filtering version 2020-10
    if camera == 'Kiruna2' or  camera == 'Kiruna3':  # (1) for SONY camera
        Gmin_c1, Gmax_c1, Rratio_c1, Bratio_c1 = 0.28, 0.79, 0.99, 0.99  # G= 73-200 (0.28-0.79), R/G>0.99 or B/G>0.99
        Gmin_c2, Gmax_c2, Rratio_c2, Bratio_c2 = 0.15, 0.43, 0.85, 1.82  # G= 59-109 (0.23-0.43), R/G>0.86, (R+B)/G>1.82
        Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c = 0.56, 0.19, 0.00, 0.30, 0.20, 0.80   #  Cloud
        return(Gmin_c1,Gmax_c1,Rratio_c1,Bratio_c1,  Gmin_c2,Gmax_c2,Rratio_c2,Bratio_c2, Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c)
        #Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1, Gmin_c2,Gmax_c2,RGratio_c2,BGratio_c2, Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = cloudcriterion(camera)
    
    else:  # (2) for Nikon camera
        Gmin_c1, Gmax_c1, Rratio_c1, Bratio_c1 = 0.39, 0.66, 0.97, 0.60  # G=100-170 (0.39-0.66), R/G>0.97, B/G=all
        Gmin_c2, Gmax_c2, Rratio_c2, Bmax_c2 = 0.21, 0.43, 1.10, 0.12  # G= 55-110 (0.21-0.43), R/G>1.10, B<30 (0.12)
        Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c  = 0.00, 0.16, 0.10, 0.8,  0.15, 0.83    #  Cloud
        if camera == 'Abisko': Smin_c,  Smax_c  = 0.00, 1.0   #  Cloud
        return(Gmin_c1,Gmax_c1,Rratio_c1,Bratio_c1,  Gmin_c2,Gmax_c2,Rratio_c2,Bmax_c2, Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c)
            #Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1,  Gmin_c2,Gmax_c2,RGratio_c2,Bmax_c2,  Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c = cloudcriterion(camera)


#------------------------------------
# used in derive_mask() program as
#  maskW,maskA,maskS, cntMaybe,cntYes,cntStrong = aurorafilter(H,L,S, R,G,B, camera, filter_level)
def aurorafilter(H,L,S, R,G,B, camera, filter_level):
    #  --- python3 function ---
    # input: L[*,*],H,S, R,G,B camera ('Kiruna' or 'Abisko')
    #               (this is output of H,L,S = RGB_everything_HLS(AllSkyPicture)
    #        .... = HLS_max_min(camera)
    #        filter_level = *0**: HLS
    #        filter_level = *1**: RGB
    #--------#---------#---------#---------#---------#---------#---------#
    # output: filtered matrix for artifitial light
    #     maskW[*,*],maskA[*,*],maskS[*,*]: filter
    #               1 = True  = light (to be masked)
    #               0 = False = clear sky
    
    filter_level = int(filter_level) % 10
    mask0,maskW,maskA,maskS,maskN = np.zeros(R.shape,dtype=bool), np.zeros(R.shape,dtype=bool), np.zeros(R.shape,dtype=bool), np.zeros(R.shape,dtype=bool), np.zeros(R.shape,dtype=bool)   # An "False"-array with the same dimension as R

    # (1) RGB method (only SONY camera)
    if (camera == 'Kiruna2') and filter_level == 1:
        mask0   = (H>0.19) & (H<0.48) & (G>=0.40) & (R<0.97) & (R<G*0.98) # green range including diffuse aurora
        maskW   = (mask0 == True) & ( (G>=0.31) & (R<0.65) & (B>=0.19) & (B<0.94) ) # weak aurora
        maskA   = (mask0 == True) & ( (G>=0.71) & (R<0.80) & (B>=0.33) & (B<0.94) ) # aurora arc
        maskS   = (mask0 == True) & ( (G>=0.85) & (R<0.97) & (B>=0.52) & (B<0.94) ) # strong aurora
        maskN   = maskN == True

    elif (camera == 'Kiruna3') and filter_level == 1:
        mask0   = (G>=0.40) & (H>=0.19) & (H<0.48) & (R<G*0.95) & (R+B<G*1.90) # green range including diffuse aurora

        #2020c maskS   = ( (G>=0.99) & (R<G*0.98) & (B<G*0.95) ) | ( (G>=0.86) & (R<G*0.78) & (B<G*0.78) ) | ( (G>=0.80) & (R<G*0.55) & (B<G*0.55) ) # strong aurora
        #2020c maskA   = ( (G>=0.51) & (G<0.64) & (R<G*0.92) & (B<G*0.73) ) | ( (G>=0.62) & (G<0.90) & (R<G*0.90) & (B<G*0.90) )  # aurora arc
        #2020c maskW   = ( (G<0.64) & (B<G*0.99) ) | (R<G*0.79) | ( (G>=0.70) & (R<G*0.92) ) # weak aurora

        maskN   = (G>=0.97) & (H>=0.25) & (H<0.34) & (R>=G*0.50) & (R<G*0.95) &  ((G>=0.995) | (R<G*0.92))   # nitrogen line (pink color) is mixed

        maskS1  = (G>=0.95) &  (H>=0.26) & (H<0.36) & (R<G*0.75)  # strong aurora without light-effect
        maskS2  = (G>=0.88) &  (H>=0.30) & (H<0.36) & (R<G*0.70) & (B<G*0.66)  # strong aurora without light-effect
        maskS3  = (G>=0.83) &  (H>=0.30) & (H<0.34) & (R<G*0.62) & (B<G*0.57)   # strong aurora without light-effect
        maskS4  = (G>=0.76) &  (H>=0.32) & (H<0.36) & (R<G*0.43) & (B<G*0.50) & (R+B<G*0.89)   # strong aurora with without light-effect
        maskS5  = (G>=0.77) & (G<0.83) & (H>=0.30) & (H<0.352) & (R<G*0.60) & (B<G*0.64) & (R+B>=G*0.96)   # strong aurora without light-effect
        maskS6  = (G>=0.75) & (G<0.95) & (H>=0.26) & (H<0.31) & (R<G*0.69) & (B<G*0.54)   # strong aurora without light-effect
        maskS   = (maskN == True) | (maskS1 == True) | (maskS2 == True) | (maskS3 == True) | (maskS4 == True) | (maskS5 == True) | (maskS6 == True)
        
        maskA1  = (G>=0.69) &  (H>=0.32) & (H<0.36) & (R<G*0.47) & (B<G*0.51)   # aurora arc/strong diffuse without light-effec
        maskA2  = (G>=0.69) &  (H>=0.36) & (H<0.41) & (R<G*0.70) & (B<G*0.80)   # aurora arc/strong diffuse without light-effec
        maskA3  = (G>=0.65) & (G<0.80) & (H>=0.28) & (H<0.35) & (R<G*0.70) & ((B<G*0.62) | (R+B<G*1.21))  #  aurora arc/strong diffuse without light-effect
        maskA4  = (G>=0.59) & (G<0.70) & (H>=0.24) & (H<0.27) & (R<G*0.80) & (B<G*0.55)   #  aurora arc/strong diffuse without light-effect
        maskA5  = (G>=0.59) & (G<0.77) & (H>=0.20) & (H<0.24) & (R<G*0.92) & (B<G*0.65)   #  aurora arc/strong diffuse without light-effect
        maskA6  = (G>=0.55) & (G<0.68) & (H>=0.30) & (H<0.36) & (R<G*0.61) & (B<G*0.61)   #  aurora arc/strong diffuse without light-effect
        maskA7  = (G>=0.50) & (G<0.69) & (H>=0.25) & (H<0.35) & (R<G*0.75) & (B<G*0.51)   #  aurora arc/strong diffuse without light-effect
        maskA   = (maskS == True) | (maskA1 == True) | (maskA2 == True) | (maskA3 == True) | (maskA4 == True) | (maskA5 == True) | (maskA6 == True) | (maskA7 == True)
        
        maskW0  =  (G>=0.85) & (G<0.94) & (H>=0.31) & (H<0.38) & (R<G*0.92) & (B<G*0.93) & (R+B<G*1.83) # light-contamination
        maskW1  =  (G>=0.61) & (G<0.87) & (H>=0.30) & (H<0.36) & (R<G*0.83) & (B<G*0.80) & (R+B<G*1.60) # weak aurora
        maskW2  =  (G>=0.61) & (G<0.80) & (H>=0.36) & (H<0.43) & (R<G*0.88) & (B<G*0.92) & (R+B<G*1.70) # light-contamination
        maskW3  =  (G>=0.40) & (G<0.80) & (H>=0.24) & (H<0.31) & (R<G*0.72) & (B<G*0.40)   #  aurora arc/strong diffuse without light-effect
        maskW4  =  (G>=0.40) & (G<0.65) & (H>=0.33) & (H<0.38) & (R<G*0.87) & (B<G*0.90) & (R+B<G*1.70) & (R+G+B>0.98) # weak aurora
        maskW5  =  (G>=0.40) & (G<0.65) & (H>=0.40) & (H<0.47) & (R<G*0.72) & (B<G*0.96) & (R+B<G*1.60) & (R+G+B>0.98) # weak aurora

        maskW6  =  (G>=0.56) & (G<0.75) & (H>=0.18) & (H<0.21) & (R<G*0.95) & (B<G*0.64) # aurora through cloud (may skip this)

        maskW   = (maskA == True) | (maskW0 == True) | (maskW1 == True) | (maskW2 == True) | (maskW3 == True) | (maskW4 == True) | (maskW5 == True) | (maskW6 == True)
        #maskC1 = ((G>=0.43) & (G<0.64) & (H>=0.19) & (H<0.24) & (R>=G*0.84) & (B>=G*0.51) & (B<G*0.66))
        #maskC2 = ((G>=0.46) & (G<0.64) & (H>=0.25) & (H<0.28) & (R>=G*0.67) & (R<G*0.77) & (B>=G*0.52) & (B<G*0.64))
        #maskL1 = ((G>=0.65) & (G<0.92) & (H>=0.36) & (H<0.67) & (R>=G*0.88) & (R<G*1.01) & (B>=G*0.92) & (B<G*1.05))
        #maskL2 = ((G>=0.68) & (G<0.74) & (H>=0.15) & (H<0.28) & (R>=G*0.88) & (R<G*0.94) & (B>=G*0.92) & (B<G*1.00))
    
    
        maskN   = (mask0 == True) & (maskN == True)
        maskS   = (mask0 == True) & (maskS == True)
        maskA   = (mask0 == True) & (maskA == True)
        maskW   = (mask0 == True) & (maskW == True)

    # (2) HSL method (both SONY and Nikon cameras)
    else:
        Hmin_w, Hmax_w, Smin_w, Smax_w, Lmin_w, Lmax_w, Hmin_a, Hmax_a, Smin_a, Smax_a, Lmin_a, Lmax_a, Hmin_s, Hmax_s, Smin_s, Smax_s, Lmin_s, Lmax_s, Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = HLS_max_min(camera)
        maskW   = (H>=Hmin_w) & (H<Hmax_w) & (S>=Smin_w) & (S<Smax_w) & (L>=Lmin_w) & (L<Lmax_w) # weak aurora
        maskA   = (H>=Hmin_a) & (H<Hmax_a) & (S>=Smin_a) & (S<Smax_a) & (L>=Lmin_a) & (L<Lmax_a) # aurora arc
        maskS   = (H>=Hmin_s) & (H<Hmax_s) & (S>=Smin_s) & (S<Smax_s) & (L>=Lmin_s) & (L<Lmax_s) # strong aurora



    cntMaybe,cntYes,cntStrong  = np.count_nonzero(maskW),np.count_nonzero(maskA),np.count_nonzero(maskS)

    return(maskW,maskA,maskS, cntMaybe,cntYes,cntStrong)


#------------------------------------
# used in derive_mask() program as
#  maskC, cntCloud = cloudfilter(H,L,S, R,G,B, camera, filter_level)
def cloudfilter(H,L,S, R,G,B, camera, filter_level):
    #  --- python3 function ---
    # input: L[*,*],H,S, R,G,B camera ('Kiruna' or 'Abisko')
    #        .... = cloudcriterion(camera)
    #     filter_level = 0: to get maskC_core
    #     filter_level = 1: to get maskC_full => actually used
    #--------#---------#---------#---------#---------#---------#---------#
    # output: filtered matrix for artifitial light
    #     maskC_raw[*,*]:
    #               1 = True  = light (to be masked)
    #               0 = False = clear sky
    
    maskC_raw = np.zeros(R.shape,dtype=bool)  # An "False"-array with the same dimension as R

    #Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1,  Gmin_c2,Gmax_c2,RGratio_c2,Bmax_c2,  Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c = cloudcriterion(camera)
    
    # (1) SONY camera
    '''
    if camera == 'Kiruna2' or  camera == 'Kiruna3':
        Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1, Gmin_c2,Gmax_c2,RGratio_c2,BGratio_c2, Hmin_c, Hmax_c, Smin_c, Smax_c, Lmin_c, Lmax_c = cloudcriterion(camera)
        maskC_raw  = (G>=Gmin_c1) & (G<Gmax_c1) & ( (R>=G*RGratio_c1) | (B>=G*BGratio_c1) )  #filter clear cloud
        if filter_level != 0: maskC_raw  = (maskC_raw == True) | ( (G>=Gmin_c2) & (G<Gmax_c2) & (R>=G*RGratio_c2) & (R+B>=G*BGratio_c2) ) #filter all cloud
    '''
    if camera == 'Kiruna2':
        #2020b maskC_raw  = (G>=0.28) & (G<0.79) & ( (R>=G*0.99) | (B>=G*0.995) )  # clear cloud: G= 73-200 (0.28-0.79), R/G>0.99 or B/G>0.99
        #2020b if filter_level != 0: maskC_raw  = (maskC_raw == True) | ( (G>=0.15) & (G<0.43) & (R>=G*0.85) & (R+B>=G*1.82) ) # + weak cloud: G= 59-109 (0.23-0.43), R/G>0.86, (R+B)/G>1.82
        maskC_raw  = (G>=0.28) & (G<0.79) & (R>=G*0.99) & (B>=G*0.66)
        if filter_level != 0: maskC_raw = (maskC_raw == True) | ( (G>=0.23) & (G<0.43) & (R>=G*0.85) & (B>=G*0.86) )
    
    elif camera == 'Kiruna3':
        maskC1 = (G>=0.995) & (R>=0.995) & (B>=G*0.92) & (B<G-0.018) & (B<R-0.018)   # cloud with town light R,G>=254,
        maskC2 = (G>=0.94) & (G<0.995) & (H>=0.10) & (H<0.24) & (R>=0.99)  & (B>=G*0.89) & (B<G-0.018)   # cloud with town light
        maskC3 = (G>=0.60) & (G<0.96) & (H>=0.10) & (H<0.19) & (R>=G*0.97) & (R<G*1.10) & (B>=G*0.65) & (B<G*0.90) # cloud with town light
        maskC4 = (G>=0.64) & (G<0.77) & (H>=0.18) & (H<0.27) & (R>=G*0.92) & (R<G*1.00) & (B>=G*0.86) & (B<G*0.96) # cloud near moon
        maskC5 = (G>=0.50) & (G<0.67) & (H>=0.10) & (H<0.20) & (R>=G*0.95) & (B>=G*0.55) & (B<G*0.80) & (R+B>=G*1.52)   # cloud (less blight)
        maskC6 = (G>=0.38) & (G<0.55) & (H>=0.12) & (H<0.20) & (R>=G*0.92) & (B>=G*0.50) & (B<G*0.70) & (R+B>=G*1.49)   # cloud (less blight)
        maskC7 = (G>=0.41) & (G<0.60) & (H>=0.17) & (H<0.29) & (R>=G*0.65) & (B>=G*0.50) & (B<R-0.10)   # cloud reflecting aurora
        
        maskC_raw   = (maskC3 == True) | (maskC4 == True) | (maskC5 == True) | (maskC6 == True) | (maskC7 == True)
        if filter_level != 0: maskC_raw   = (maskC1 == True) | (maskC2 == True) | (maskC_raw == True)  # include extra blight pixel by town light
        
        maskC8 = (G>=0.60) & (G<0.70) & (H>=0.26) & (H<0.32) & (R>=G*0.90) & (R<G*0.97) & (B>=G*0.88) # cloud under diffuse aurora
        maskC9 = (G>=0.55) & (G<0.70) & (H>=0.47) & (H<0.55) & (R>=G*0.80) & (R<G*0.95) & (B>=G*0.95) # cloud under diffuse aurora

        if filter_level != 0: maskC_raw  = (maskC_raw == True) | (maskC8 == True) | (maskC9 == True)

    # (2a) Nikon camera, with RGB method
    elif camera == 'Kiruna0':
        Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1,  Gmin_c2,Gmax_c2,RGratio_c2,Bmax_c2,  Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c = cloudcriterion(camera)
        maskC_raw  = (G>=Gmin_c1) & (G<Gmax_c1) & (R>=G*RGratio_c1)  # filter clear cloud
        maskC_raw  = (maskC_raw == 1) | ( (G>=Gmin_c2) & (G<Gmax_c2) & (R>=G*RGratio_c2) & (B<Bmax_c2) ) #filter all cloud
        '''
        maskC_raw  = (G>=0.39) & (G<0.66) & ( (R>=G*0.99) & (B>=G*0.99) ) | ( (G>=0.21) & (G<0.43) & (R>=G*1.10) & (B<0.12) )
        # clear cloud: G=100-170 (0.39-0.66), R/G>0.99, B/G>0.99
        # + weak cloud: G= 55-110 (0.21-0.43), R/G>1.10, B<30 (0.12)
        '''
        # old maskC_raw  = ((G>=0.39) & (G<0.66) & (R>=G*0.97) & (B>=G*0.60)) | ((G>=0.21) & (G<0.43) & (R>=G*1.10) & (B<0.12))


    # (2b) Nikon camera, with HLS method
    else:
        '''
        Gmin_c1,Gmax_c1,RGratio_c1,BGratio_c1,  Gmin_c2,Gmax_c2,RGratio_c2,Bmax_c2,  Hmin_c,  Hmax_c,  Smin_c,  Smax_c,  Lmin_c,  Lmax_c = cloudcriterion(camera)
        maskC_raw   = (H>=Hmin_c) & (H<Hmax_c) & (S>=Smin_c) & (S<Smax_c) & (L>=Lmin_c) & (L<Lmax_c) # cloud fileter
        '''
        maskC_raw  = (H<0.16) & (S>=0.10) & (S<0.80) & (L>=0.15) & (L<0.83)                         # for Kiruna
        if camera == 'Abisko': maskC_raw = (H<0.16) & (S>=0.00) & (S<1.00) & (L>=0.15) & (L<0.83)   # for Abisko
        # no change

    cntCloud_raw  = np.count_nonzero(maskC_raw)      # counting numbers of True (=1) = cloud pixel (False = 0)
    return(maskC_raw, cntCloud_raw)


#------------------------------------
# used in derive_mask() program as
#  maskL, cntLight = lightfilter(H,L,S, R,G,B, camera)
def lightfilter(H,L,S, R,G,B, camera):
    #  --- python3 function ---
    # input: L[*,*],H,S, R,G,B camera ('Kiruna' or 'Abisko')
    #        .... = RGB_mooncriterion(camera)
    #--------#---------#---------#---------#---------#---------#---------#
    # output: maskL[*,*]: filtered matrix for artifitial light
    #               1 = True  = light (to be masked)
    #               0 = False = clear sky
    
    maskL = np.zeros(R.shape,dtype=bool)   # An "False"-array with the same dimension as L
    xdim,ydim = L.shape # 430x430 is default, but y (holizontal) might be trimed more to remove city light
    #       x = vertical downward (north = 0, south = max)
    #       y = holizontal rightward (east = 0, west = town = max)
    #-------------------
    # (1) SONY camera
    '''
    if camera == 'Kiruna2' or  camera == 'Kiruna3':
        Gmin_m1,Gmax_m1,RGratio_m1,BGratio_m1,  Gmin_m2,Gmax_m2,RGratio_m2,BGratio_m2,  Gmin_m3, Gmax_m3, RGratio_m3, RGratio_m4, Gmin_l1,Gmax_l1,RGratio_l1,BGratio_l1, Gmin_l2,Gmax_l2,RGratio_l2,BGratio_l2 = RGB_mooncriterion(camera)
        maskL  = ( (G>=Gmin_l1) & (G<Gmax_l1) & (R>=G*RGratio_l1) & (B>=G*BGratio_l1) ) | ( (G>=Gmin_l2) & (G<Gmax_l2) & (R>=G*RGratio_l2) & (B < G*BGratio_l2) ) #filter light from north
        '''
    if camera == 'Kiruna2':
        #2020b maskL  = ( (G>=0.68) & (G<0.95) & (R>=G*0.83) & (B>=G*0.71) ) | ( (G>=0.62) & (G<0.69) & (R>=G*0.73) & (B < G*0.65) )
        maskL = ( (G>=0.68) & (G<0.95) & (R>=G*1.03) & (B<G*0.83) ) | ( (G>=0.62) & (G<0.69) & (R>=G*0.73) & (R<G*0.96) & (B < G*0.65) ) |  ( (G>=0.68) & (G<0.95) & (R>=G*0.95) & (R<G*1.03) & (B>=G*0.70) & (B<G*0.83) )  #  light from north (only until 2020-10)
    
    elif camera == 'Kiruna3':
        maskL1 = ((G>=0.65) & (G<0.92) & (H>=0.45) & (H<0.67) & (R>=G*0.91) & (R<G*1.01) & (B>=G*0.92) & (B<G*1.05)) # town light
        maskL2 = ((G>=0.66) & (G<0.74) & (H>=0.39) & (H<0.50) & (R>=G*0.88) & (R<G*0.94) & (B>=G*0.92) & (B<G*1.00) & (R+B>=G*1.82)) # town light
        maskL3 = ((G>=0.66) & (G<0.74) & (H>=0.15) & (H<0.19) & (R>=G*0.94) & (R<G*1.01) & (B>=G*0.96) & (B<G*1.00)) # town light
        maskL   = (maskL1 == True) | (maskL2 == True) | (maskL3 == True)
        
        maskL5 = ((G>=0.71) & (G<0.92) & (H>=0.26) & (H<0.46) & (R>=G*0.91) & (R<G*0.98) & (B>=G*0.92) & (B<G*0.99)) # town light with diffuse aurora
        maskL   = (maskL == True) | (maskL5 == True)

    # (2a) Nikon camera, using RGB method
    else:
        pass

    cntLight = np.count_nonzero(maskL)     # counting numbers of True (False = 0) = moon pixel
    return(maskL, cntLight)


#------------------------------------
# used in derive_mask() program as
#  maskM_core, maskM_blight, maskM_full, cntMoon_core,cntMoon_blight,cntMoon_full
#       = moonfilter(H,L,S, R,G,B, camera, filter_level)
def moonfilter(H,L,S, R,G,B, camera, filter_level):
    #  --- python3 function ---
    # input: L[*,*],H,S, R,G,B camera ('Kiruna' or 'Abisko')
    #        .... = RGB_mooncriterion(camera)
    # filter_level  = **0* - **4*: moon filter (core part + without/with surrounding part)
    #               = **5* - **9*: + filling boundaries
    #--------#---------#---------#---------#---------#---------#---------#
    # output: filtered matrix for artifitial light
    #     maskM_core[*,*]  : just core part  (**_m1)
    #     maskM_blight[*,*]: just core+nearby part  (**_m1, **_m2)
    #     maskM_full[*,*]  : including surrounding part (**_m1, **_m2, **_m3)
    #               1 = True  = light (to be masked)
    #               0 = False = clear sky

    maskM_core,maskM_blight,maskM_full = np.zeros(L.shape,dtype=bool),np.zeros(L.shape,dtype=bool),np.zeros(L.shape,dtype=bool)  # An "False"-array with the same dimension as L
    
    #-------------------
    # (1) using RGB method
    '''
    if camera == 'Kiruna2' or  camera == 'Kiruna3':
        Gmin_m1,Gmax_m1,RGratio_m1,BGratio_m1,  Gmin_m2,Gmax_m2,RGratio_m2,BGratio_m2,  Gmin_m3, Gmax_m3, RGratio_m3, RGratio_m4, Gmin_l1,Gmax_l1,RGratio_l1,BGratio_l1, Gmin_l2,Gmax_l2,RGratio_l2,BGratio_l2 = RGB_mooncriterion(camera)
        maskM_core   = (G>=Gmin_m1) & (G<Gmax_m1) & (R>=G*RGratio_m1) & (R+B>=G*BGratio_m1)  # moon filter (core)
        maskM_blight = (maskM_core == True) | ( (G>=Gmin_m2) & (G<Gmax_m2) & (R>=G*RGratio_m2) & (B>=G*BGratio_m2) )   # moon filter (core + nearby)
        maskM_full   = (maskM_blight == True) | ( (G>=Gmin_m3) & (G<Gmax_m3) & (R>=G*RGratio_m3) & (R < G*RGratio_m4) )   # + moon filter (both core and surrounding)
        '''
    if camera == 'Kiruna2':
        #2020b maskM_core   = ( (G>=0.94) & (R>=G*0.99) & (R+B>=G*1.98) )  # moon filter (core): G>240 (>0.94), (R/G>0.99 or (R+B)/G>1.98)
        #2020b maskM_blight = ( (G>=0.64) & (R>=G*0.98) & (B>=G*0.96) ) | (maskM_core == True)  # + moon_nearby: G>185 (>0.64), R/G ratio>0.98, B/G>0.96
        #2020 maskM_full   : no change
        maskM_core   = ( (G>=0.96) & (R>=G*0.99) & (B>=G*0.975) )
        maskM_blight = ( (G>=0.72) & (R>=G*0.98) & (B>=G*0.96) ) | (maskM_core == True)
        maskM_full   = ( (G>=0.40) & (G<0.80) & (R>=G*0.98) & (R < G*1.05) ) | (maskM_blight == True)  # + moon_surrounding: G>101 (>0.40), R/G ratio=0.98-1.05
    
    elif camera == 'Kiruna3':
        maskM_254    = (G>=0.995) & (R>0.995) & (B>=G-0.024) & (B<G+0.005)   # moon filter (core1): R>=G>=254 & B>=G-4 or
        maskM_core   = ((G>=0.95) & (G<0.997) & (R>=G-0.005) & (R<G+0.017) & (B<=G) & (R+B>=G*1.95) & (R+B<G*2.01)) | (maskM_254 == True) # moon filter (core2): G>240, (R+B)/G>1.98)
        maskM_blight = ((G>=0.70) & (G<0.97) & (R>=G*0.994) & (R<G*1.024) & (B>G*0.93) & (B<G*0.98)) | (maskM_core == True)  # + moon_nearby: R+-=G>=179 & B+-=G*0.95
        maskM_full   = ((G>=0.60) & (G<0.90) & (H>=0.42) & (H<0.50) & (R>=G*0.79) & (R<G*0.98) & (B>=G*0.94) & (B<G-0.005)) | (maskM_blight == True)  # + moon_nearby: G>185 (>0.64), R/G ratio>0.98, B/G>0.96

    # (2a) Nikon camera, with RGB method
    elif camera == 'Kiruna0':
        Gmin_m1,Gmax_m1,RGratio_m1,BGratio_m1,  Gmin_m2,Gmax_m2,RGratio_m2,BGratio_m2 = RGB_mooncriterion(camera)
        maskM_core   = (G>=Gmin_m1) & (G<Gmax_m1) & (R>=G*RGratio_m1) & (B>G*BGratio_m1)  # moon filter (core part)
        maskM_blight = (maskM_core == True) | ( (G>=Gmin_m2) & (G<Gmax_m2) & (R>=G*RGratio_m2) & (B>G*BGratio_m2) ) # moon filter (both core and surrounding)
        maskM_full   = maskM_blight[:,:]
        '''
        maskM_core   = ( (G>=0.92) & (R>=G*0.99) & (B>=G*0.97) )  # moon filter (core): G>200 (>0.78), R/G ratio>0.93, B/G>0.86
        maskM_blight = ( (G>=0.64) & (G<0.92) & (R>=G*0.97) & (B>=G*0.95) ) | (maskM_core == True)  # + moon_nearby: G=170-200 (0.66-0.78), R/G>0.97, B/G>0.97
        maskM_full   = maskM_blight[:,:]
        '''
        # no change

    # (2b) Nikon camera, with HLS method, but this NOT be used
    else:
        Hmin_m1, Hmax_m1, Smin_m1, Smax_m1, Lmin_m1, Lmax_m1, Hmin_m2, Hmax_m2, Smin_m2, Smax_m2, Lmin_m2, Lmax_m2 = HLS_mooncriterion(camera)
        maskM_core   = (H>=Hmin_m1) & (H<Hmax_m1) & (S>=Smin_m1) & (S<Smax_m1) & (L>=Lmin_m1) & (L<Lmax_m1) # moon filter (core part)
        maskM_blight = (maskM_core == True) | ((H>=Hmin_m2) & (H<Hmax_m2) & (S>=Smin_m2) & (S<Smax_m2) & (L>=Lmin_m2) & (L<Lmax_m2)) # moon filter (both core and surrounding)
        maskM_full   = maskM_blight[:,:]
        '''
        maskM_core   = (S<0.20) & (L>=0.90)  # moon filter (core part)
        maskM_blight = (maskM_core == True) | ( (S<0.13) & (L>=0.80)  # moon filter (both core and surrounding)
        maskM_full   = maskM_blight[:,:]
         '''
        # no change

    #-------------------
    # option 2: lower threshold for connected area (in additon to option 0)
    #    --> define  M[x,y] = 1 for moon (not twilight), = 0 for dark sky
    if int(filter_level/10) % 10 >= 5:
        xdim, ydim = L.shape  # 455x455=207025  or 446x446=198916
        for x in range(1,xdim-1):
            for y in range(1,ydim-1):
                mask4pt = maskM_core[x-1,y] + maskM_core[x+1,y] + maskM_core[x,y-1] + maskM_core[x,y+1]
                mask4pt2 = maskM_blight[x-1,y] + maskM_blight[x+1,y] + maskM_blight[x,y-1] + maskM_blight[x,y+1]
                mask4pt3 = maskM_full[x-1,y] + maskM_full[x+1,y] + maskM_full[x,y-1] + maskM_full[x,y+1]
                mask8pt = maskM_core[x-1,y-1] + maskM_core[x+1,y+1] + maskM_core[x+1,y-1] + maskM_core[x-1,y+1] + mask4pt
                mask8pt2 = maskM_blight[x-1,y-1] + maskM_blight[x+1,y+1] + maskM_blight[x+1,y-1] + maskM_blight[x-1,y+1] + mask4pt
                mask8pt3 = maskM_full[x-1,y-1] + maskM_full[x+1,y+1] + maskM_full[x+1,y-1] + maskM_full[x-1,y+1] + mask4pt
                if mask4pt ==4 or mask8pt >=6: maskM_core[x,y] = 1
                if mask4pt2 ==4 or mask8pt2 >=6: maskM_blight[x,y] = 1
                if mask4pt3 ==4 or mask8pt3 >=6: maskM_full[x,y] = 1
    else: pass

    cntMoon_core,cntMoon_blight,cntMoon_full = np.count_nonzero(maskM_core),np.count_nonzero(maskM_blight),np.count_nonzero(maskM_full) # counting numbers of True (False = 0) = moon pixel
    return(maskM_core, maskM_blight, maskM_full, cntMoon_core,cntMoon_blight,cntMoon_full)


#------------------------------------
#  M,M2,cntMoon0,cntInside,cntMoon2,Xcenter0,Ycenter0,
#      Xcenter_local,Ycenter_local,Xcenter2,Ycenter2, radiusL,new_radius,flag
#    = judge_ASC.mooncenter(H,L,S, R,G,B, maskM_core, maskM_blight, maskM_full, camera, radius, filter_level)
#
def mooncenter(L, maskM_core, maskM_blight, maskM_full, camera, radius, filter_level):
    # input:   maskM_core, maskM_blight, maskM_full = minimum, medium, maximum level of moon mask
    #          filter_level = **0*: use maskM_core
    #                       = **1*: use maskM_blight
    #                       = **2*: use maskM_full
    #
    M,cntMoon0,cnt_local,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2, radiusL,new_radius,flag = mooncenter2(L, maskM_core, camera, radius, radius)
    # flag = 0-10

    if flag >= 900 or (flag > -20 and flag <= -10):
        new_radius = max(radiusL-5,new_radius-5)
        new_radius = max(new_radius,10)
        M3,cntMoon3,cnt_local3,cntMoon4,Xcenter_mask3,Ycenter_mask3,XcenterL3,YcenterL3,Xcenter4,Ycenter4, radiusL2,new_radius2,flag3 = mooncenter2(L, maskM_core, camera, radius, new_radius)
        if flag3 >= 100:
            new_radius = new_radius2
            radiusL = radiusL2
            flag = flag3 + 4000
        elif flag3 >= 0:
            new_radius = new_radius2
            radiusL = radiusL2
            M = M3
            flag = flag3 + 20
        else:
            radiusL = 0
            flag = flag3 - 4000

        if flag3 >= 0: cntMoon0,cnt_local,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2 = cntMoon3,cnt_local3,cntMoon4,Xcenter_mask3,Ycenter_mask3,XcenterL3,YcenterL3,Xcenter4,Ycenter4


    if flag >= 0: radius = new_radius
    

    filter_level = int(filter_level/10) % 10
    if filter_level == 0: maskM  =  maskM_core
    elif filter_level == 1: maskM  =  maskM_blight
    else: maskM = maskM_full
    
    xdim,ydim = L.shape # 455x455=207025  or 446x446=198916
    area = min(radius*radius*5, 2500)       # about x 2.2 (50x50)
    area2 = min(radius*radius*25, 10000)    # about x 5   (100x100)
    area3 = min(radius*radius*200, 60000)  # about x 14   (240x240)
    M2 = np.zeros(L.shape,float)   # Define M as an array with the same dimension as L
    if flag >= 0 and flag <= 299:
        for x in range(0,xdim):
            for y in range(0,ydim):
                distance2 = (x-Xcenter2)*(x-Xcenter2) + (y-Ycenter2)*(y-Ycenter2)
                if distance2 < area:
                    M2[x,y] = 1
                elif distance2 < area2:
                    M2[x,y] = maskM_full[x,y]
                elif distance2 < area3:
                    M2[x,y] = maskM_full[x,y]
        cntMoon2 = np.count_nonzero(M2)      # 1=moon, 0=otherswise --> moon count

    return(M,M2,cntMoon0,cnt_local,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2, radiusL,new_radius,flag)


#------------------------------------
# used only in "mooncenter()"
def mooncenter2(L, maskM, camera, radius_judge, radius):
    #  --- python3 function ---
    # Identify the moon or star and find center location.
    #      note: moon size in 450x450 pixel image is abour Diameter 35-40 pixels
    #
    # input:  L[*,*,*], S[*,*,*]: L and S values in HLS color code
    #         HLS_mooncriterion(camera) &  RGB_mooncriterion(camera): threshold to judge each pixel as moon or not
    #         radius: radius of moon/star in pixels
    #         maskM, maskL, maskC:  mask(Treu=1, False=0) for moon, human light, cloud.
    #
    #   *,*,*,,,, = mooncenter(H,L,S, R,G,B, camera, radius):
    # output: M[*,*,*]: Moon filter (moon=1, otherswise=0)
    #         cntMoon0,cntMoon,cnt_local: numbers of moon-pixels, identified by
    #               mask(cntMoon0), M(cntMoon), and small subregion (cnt_local)
    #         Xcenter_mask,Ycenter_mask,Xcenter2,Ycenter2,XcenterL,YcenterL:
    #               center location in pixel (i.e., numpy array location), by
    #              mask(***0), M(no mark), and small subregion (***1)
    #         flag: -8 or -9 = moon count is too little,  -99 = error
    #               0 or -1 or -2 or -3 = probably correct
    #                  0 or -1 = center locations are similar in 2 methods
    #                  -2 = similar center location after enlarging subregion
    #                  -3 = XYcenter \sim XYcenterL  after enlarging  subregion
    #               could not identify, -6 = at the edge,
    #               -4 or -5 = could be correct but not sure
    #               -6 = others


    Xcenter_mask, Ycenter_mask = 0, 0
    XcenterL, YcenterL = 0,0
    Xcenter2, Ycenter2 = 0, 0
    xdim,ydim = L.shape # 455x455=207025  or 446x446=198916
    M = np.zeros(L.shape,float)   # Define M as an array with the same dimension as L
    # alternativ copy method: M2=M[:,:] or M2=M.copy()

    #----------------------
    ## (99, 98) No moon: Number of moon pixles is 0 or <10
    #    &
    ## (method-A) Simplest way to estimate the center location using maskM
    #
    cntMoon0 = np.count_nonzero(maskM)      # True=1=moon, 0=otherswise --> moon count
    if cntMoon0 == 0:
        return(M,0,-1,-1,-1,-1,-1,-1,-1,-1,radius,0,-99)
    else:
        for x in range(xdim):
            for y in range(ydim):
                Xcenter_mask = Xcenter_mask + x*maskM[x,y]
                Ycenter_mask = Ycenter_mask + y*maskM[x,y]
        Xcenter_mask, Ycenter_mask = int((Xcenter_mask/cntMoon0)+0.5), int((Ycenter_mask/cntMoon0)+0.5)

    if cntMoon0 < 10:
        return(M,cntMoon0,-1,-1,Xcenter_mask,Ycenter_mask,-1,-1,-1,-1,radius,0,-98)

    area_max = cntMoon0/3.14
    radius_max = int(math.sqrt(area_max)) + 1
    if radius_max < radius-2:
        radius = radius-2
    elif radius_max < radius-1:
        radius = radius-1

    radius_p = radius+1
    size,area,area_judge = radius+radius_p, radius*radius,radius_judge*radius_judge
    area_local,area_Lsum = int(area*2.8), int(area*3)
    minimumCnt = area/2.
    
    #----------------------
    ## Find out dense region of "moon-like" pixels.
    ## (method-B) Center location from the most condensed moon pixel cluster
    #

    judgeL = np.zeros([size,size],float)
    cnt_local1, Linside1 = 0,0
    for x in range(radius,xdim-radius_p):
        for y in range(radius,ydim-radius_p):
            if maskM[x,y] == 1:
                mask1  = maskM[x-radius:x+radius_p, y-radius:y+radius_p]   # True(=1) = moon.  False(=0) = darkness
                cnt_local2 = np.count_nonzero(mask1)
                judgeL = L[x-radius:x+radius_p, y-radius:y+radius_p]
                Linside2 = np.sum(judgeL)
                if cnt_local2 > area_local and Linside2 > area_Lsum:
                    M[x,y] = 1
                    #
                    # -->  this makes M[:,:] value at "moon pixel with maskM=1"
                    #           = 1 if surraounded by sufficient moon pixels,
                    #           = 0 if not surrounded by sufficient moon pixels
                    #               or if maskM=0
                    if (cnt_local2 > cnt_local1) or (cnt_local2 == cnt_local1 and Linside2 > Linside1):
                        XcenterL, YcenterL = x,y
                        cnt_local1 = cnt_local2
                        Linside1 = Linside2
    #
    # -->  This makes XcenterL, YcenterL as the center of most condenced "moon pixel"
    #      And, cnt_local as the largest count of moon pixels inside circle with r=ragius
    #----------------------
    # (97) Number of local moon pixles is small
    # otherwise, Center location of local-moon
    #
    cntMoon2 = np.count_nonzero(M)      # 1=moon, 0=otherswise --> moon count
    if cntMoon2 < 10:
        return(M,cntMoon0,cnt_local1,-1,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,-1,-1,radius,0,-97)

    #----------------------
    ## (method-C) Center location from average location of moon cluster pixels (higher threshold than method-B)
    #
    #for x in range(radius,xdim-radius_p):
    #   for y in range(radius,ydim-radius_p):
    #       mask4pt = M[x-1,y] + M[x+1,y] + M[x,y-1] + M[x,y+1]
    #       mask8pt = M[x-1,y-1] + M[x+1,y+1] + M[x+1,y-1] + M[x-1,y+1] + mask4pt
    #       if mask4pt ==4 or mask8pt >=6: M[x,y] = 1
    #
    for x in range(radius,xdim-radius_p):
        for y in range(radius,ydim-radius_p):
            Xcenter2 = Xcenter2 + x*M[x,y]
            Ycenter2 = Ycenter2 + y*M[x,y]
    Xcenter2, Ycenter2 = int((Xcenter2/cntMoon2)+0.5), int((Ycenter2/cntMoon2)+0.5)   #  cntMoon2 >= 0
    #
    # -->  This makes Xcenter2, Ycenter2 as the center of most condenced "moon pixel"
    #----------------------
    # (1)-(9) Xcenter and Ycenter are most likely near the moon center after small adjustment
    #
    flag = -999
    rr12 = (XcenterL-Xcenter2)*(XcenterL-Xcenter2) + (XcenterL-Ycenter2)*(XcenterL-Ycenter2)
    rr02 = (Xcenter_mask-Xcenter2)*(Xcenter_mask-Xcenter2) + (Ycenter_mask-Ycenter2)*(Ycenter_mask-Ycenter2)
    rr01 = (Xcenter_mask-XcenterL)*(Xcenter_mask-XcenterL) + (Ycenter_mask-XcenterL)*(Ycenter_mask-XcenterL)
    if rr12 < area_judge/4 and rr02 < area_judge/4: flag = 1
    elif rr12 < area_judge/4 and rr02 < area_judge/2: flag = 2
    elif rr12 < area_judge/2 and rr02 < area_judge/4: flag = 3
    elif rr12 < area_judge/2 and rr02 < area_judge/1: flag = 4
    elif rr12 < area_judge/1 and rr02 < area_judge/1: flag = 5
    elif rr12 < area_judge/2 and rr01 < area_judge/1: flag = 6
    elif rr12 < area_judge/4: flag = 7
    elif rr12 < area_judge/2: flag = 8
    elif rr12 < area_judge/1: flag = 9
    elif rr02 < area_judge/1: flag = 10
    
    #----------------------
    # (95) Number of moon pixles / clustered moon pixles are too little, (M must be reset to 0)
    #
    if cntMoon0 < minimumCnt or cnt_local1 < minimumCnt:
        flag = -95
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2, radius,0,flag)


    #----------------------
    # Obtain moon count within a circle from the moon center for different radius
    #
    cntCircleL, area0, radiusL = moonradius(maskM, XcenterL, YcenterL, radius)
    print('moonL r(x,y)', radius, radiusL, '(', XcenterL, YcenterL, '), count', cntCircleL[5:20] )
    cntCircle2, area0, radius2 = moonradius(maskM, Xcenter2, Ycenter2, radius)
    print('moon2 r(x,y)', radius, radius2, '(', Xcenter2, Ycenter2, '), count', cntCircle2[5:20] )

    #----------------------


    if (radiusL >= radius+20 or radius2 >= radius+20):
        if radius < 19:
            if flag >= 0: flag = flag + 100
            else: flag = flag - 100
            return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2, radiusL,radius2,flag)
        else:  # cloud or daylight
            if flag >= 0: flag = flag + 1000
            else: flag = flag - 1000
            return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2, radiusL,radius2,flag)

    if radiusL == radius and radius2 == radius:
        if flag < 0 or flag > 19:
            if cntCircleL[0] < minimumCnt and cntCircle2[0] < minimumCnt: flag = -94
            else: flag = -93
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,radius,0,flag)


    if radius2 == radius or radius2+1 < radiusL: radius4 = radiusL
    else: radius4 = radius2


    sumCircleL = cntCircleL[0] + cntCircleL[1] + cntCircleL[2] + cntCircleL[3] + cntCircleL[4]
    sumCircle2 = cntCircle2[0] + cntCircle2[1] + cntCircle2[2] + cntCircle2[3] + cntCircle2[4]


    #----------------------
    # (91) Center location is at the edge
    #
    if XcenterL < radius or YcenterL < radius or XcenterL > xdim-radius or YcenterL > ydim-radius:
            return(M,cntMoon0,cnt_local1,-1,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,-1,-1,radius,0,-91)
    if Xcenter2 < radius or Ycenter2 < radius or Xcenter2 > xdim-radius or Ycenter2 > ydim-radius:
            return(M,cntMoon0,cnt_local1,-1,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,-1,-1,radius,0,-91)


    # We can repeak this by enlarging radius by using following return (if flag == 10: .....)
    #return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,10)
    #----------------------
    ## (method-D/E) Center location calculated by doubling the radius
    #
    ## (method-D) as most bright region
    #
    M3 = np.zeros(L.shape,float) # Define M as an array with the same dimension as L
    radius4p = radius4+1
    size3,area_new = radius4+radius4p, radius4+radius4
    area_local3,area_Lsum3 = int(area_new*2.8), int(area_new*3)
    minimumCnt3 = area_new/2.
    Xcenter3, Ycenter3 = 0,0
    Xcenter4, Ycenter4 = 0,0
    cnt_local3, Linside3 = 0,0
    for x in range(radius4,xdim-radius4p):
        for y in range(radius4,ydim-radius4p):
            #if L[x,y] > 0.7:
            if maskM[x,y] == 1:
                mask3   = maskM[x-radius4:x+radius4p, y-radius4:y+radius4p]   # True(=1) = moon.  False(=0) = darkness
                cnt_local4 = np.count_nonzero(mask3)
                Linside4 = np.sum(L[x-radius4:x+radius4p, y-radius4:y+radius4p])
                if cnt_local4 > area_local3 and Linside4 > area_Lsum3:
                    M3[x,y] = 1
                    if (cnt_local4 > cnt_local3) or (cnt_local4 == cnt_local3 and Linside4 > Linside3):
                        Xcenter3, Ycenter3 = x,y
                        cnt_local3 = cnt_local4
                        Linside3 = Linside4

    # -->  M3[:,:] =1 for center of moon cluster pixls, =0 otherwise
    #----------------------
    # (77) Number of moon pixles is small
    # otherwise
    #
    cntMoon4 = np.count_nonzero(M3)      # 1=moon, 0=otherswise --> moon count
    if cntMoon4 < 10:
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,radiusL,radius2,-77)

    ## (method-E) Center location from average location of moon cluster
    #
    else:
        for x in range(radius4,xdim-radius4p):
            for y in range(radius4,ydim-radius4p):
                Xcenter4 = Xcenter4 + x*M3[x,y]
                Ycenter4 = Ycenter4 + y*M3[x,y]
        Xcenter4, Ycenter4 = int((Xcenter4/cntMoon4)+0.5), int((Ycenter4/cntMoon4)+0.5)

    #
    # -->  This makes Xcenter4, Ycenter4 as the center of most condenced "moon pixel"

    #----------------------
    # (2) Xcenter and Ycenter are most likely near the moon center after small adjustment
    #
    rr24 = (Xcenter2-Xcenter4)*(Xcenter2-Xcenter4) + (Ycenter2-Ycenter4)*(Ycenter2-Ycenter4)
    rr34 = (Xcenter3-Xcenter4)*(Xcenter3-Xcenter4) + (Ycenter3-Ycenter4)*(Ycenter3-Ycenter4)
    rr04 = (Xcenter_mask-Xcenter4)*(Xcenter_mask-Xcenter4) + (Ycenter_mask-Ycenter4)*(Ycenter_mask-Ycenter4)
    rr03 = (Xcenter_mask-Xcenter3)*(Xcenter_mask-Xcenter3) + (Ycenter_mask-Ycenter3)*(Ycenter_mask-Ycenter3)
    if   rr34 < area_judge/4 and rr24 < area_judge/4: flag = 41
    elif rr34 < area_judge/4 and rr24 < area_judge/2: flag = 42
    elif rr34 < area_judge/2 and rr24 < area_judge/4: flag = 43
    elif rr34 < area_judge/2 and rr24 < area_judge/1: flag = 44
    elif rr34 < area_judge/1 and rr24 < area_judge/1: flag = 45
    elif rr34 < area_judge/2 and rr03 < area_judge/1: flag = 47
    elif rr34 < area_judge/2 and rr04 < area_judge/1: flag = 48
    elif rr34 < area_judge/1: flag = 49
    elif rr03 < area_judge/2 and rr24 < area_judge/2: flag = 51
    elif rr04 < area_judge/2 and rr24 < area_judge/2: flag = 52
    elif rr04 < area_judge/2 and rr03 < area_judge/2: flag = 53
    elif rr24 < area_judge/1: flag = 57
    elif rr04 < area_judge/1: flag = 58

    if flag >= 40 and flag <= 99:
        return(M3,cntMoon0,cnt_local3,cntMoon4,Xcenter_mask,Ycenter_mask,Xcenter3,Ycenter3,Xcenter4,Ycenter4,radiusL,radius2,flag)
    elif flag >= 0 and flag <= 39:
        return(M3,cntMoon0,cnt_local3,cntMoon4,Xcenter_mask,Ycenter_mask,Xcenter3,Ycenter3,Xcenter4,Ycenter4,radiusL,radius2,flag+400)
    
    #----------------------
    # (75) Number of moon pixles is too little, and M is reset to 0
    # (74) Number of clustered moon pixles is too little, and M is reset to 0
    #
    if cntMoon0 < minimumCnt3:
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,radiusL,radius2,-15)

    if cnt_local3 < minimumCnt3:
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,radiusL,radius2,-14)


    #----------------------
    # (71) Number of moon pixles is small
    #
    if Xcenter3 < radius4 or Ycenter3 < radius4 or Xcenter3 > xdim-radius4 or Ycenter3 > ydim-radius4:
        return(M,cntMoon0,cnt_local1,cntMoon2,Xcenter_mask,Ycenter_mask,XcenterL,YcenterL,Xcenter2,Ycenter2,radiusL,radius2,-12)
    
    #----------------------
    ## Other
    #
    return(M3,cntMoon0,cnt_local3,cntMoon4,Xcenter_mask,Ycenter_mask,Xcenter3,Ycenter3,Xcenter4,Ycenter4,radiusL,radius2,-11)

#------------------------------------
def moonradius(maskM, Xcenter, Ycenter, radius):
    # Obtain moon count within a circle from the moon center for different radius
    #   cntCircleL, area0, radiusL = moonradius(maskM, Xcenter, Ycenter, radius)
    #   cntCircle2, area0, radius2 = moonradius(maskM, Xcenter, Ycenter, radius)
    #
    xdim,ydim = maskM.shape # 455x455=207025  or 446x446=198916
    area_max = np.count_nonzero(maskM)/3.14
    cntCircle, area0 = np.zeros(20,int), np.zeros(20,int)
    for i in range(20):
        area0[i] = (radius+i+1)*(radius+i+1)
        if area0[i] > area_max: area0[i] = -1   # this makes cntCircle[i] = 0 < area0[i], where area_max = cntMoon0/3.14
    for x in range(0,xdim):
        for y in range(0,ydim):
            distance2 = (x-Xcenter)*(x-Xcenter) + (y-Ycenter)*(y-Ycenter)
            for i in range(20):
                if distance2 <= area0[i]: cntCircle[i] = cntCircle[i] + maskM[x,y]

    for i in range(20):
        area0[i] = area0[i]*3.1416
        for i in range(20):
            area0[i] = area0[i]*(0.9 + 0.003*i)  #  averaging 3.14/3
            if area0[i] <= 0: area0[i] = 2  # this makes cntCircle[i] = 0 < area0[i], where area_max = cntMoon0/3.14

    if cntCircle[0] > area0[0]:
        radius_new = radius+1
        if cntCircle[1] > area0[1]:
            radius_new = radius+2
            if cntCircle[2] > area0[2]:
                radius_new = radius+3
                if cntCircle[3]> area0[3]:
                    radius_new = radius+4
                    if cntCircle[4] > area0[4]:
                        radius_new = radius+5
                        if cntCircle[5] > area0[5]:
                            radius_new = radius+6
                            if cntCircle[6] > area0[6]:
                                radius_new = radius+7
                                if cntCircle[7] > area0[7]:
                                    radius_new = radius+8
                                    if cntCircle[8] > area0[8]:
                                        radius_new = radius+9
                                        if cntCircle[9] > area0[9]:
                                            radius_new = radius+10
                                            if cntCircle[10] > area0[10]:
                                                radius_new = radius+11
                                                if cntCircle[11] > area0[11]:
                                                    radius_new = radius+12
                                                    if cntCircle[12] > area0[12]:
                                                        radius_new = radius+13
                                                        if cntCircle[13] > area0[13]:
                                                            radius_new = radius+14
                                                            if cntCircle[14] > area0[14]:
                                                                radius_new = radius+15
                                                                if cntCircle[15] > area0[15]:
                                                                    radius_new = radius+16
                                                                    if cntCircle[16] > area0[16]:
                                                                        radius_new = radius+17
                                                                        if cntCircle[17] > area0[17]:
                                                                            radius_new = radius+18
                                                                            if cntCircle[18] > area0[18]:
                                                                                radius_new = radius+19
                                                                                if cntCircle[19] > area0[19]:
                                                                                    radius_new = radius+20

    else: radius_new = radius
    
    return(cntCircle, area0, radius_new)

#========#=========#=========#=========#=========#=========#========-#
# Part 4: calculate index using final mask and Luminocity (L) value
#  cntMaybe,cntYes,cntStrong, AverageLumS, expL_strong, LexpL_strong, LLL_strong, L_lowest,L_highest
#           = HLS_to_index(L,R,G,maskW,maskA,maskS,camera,count_loop)
#--------#---------#---------#---------#---------#---------#---------#
def HLS_to_index(L2,R,G,maskW,maskA,maskS,camera,TotalPixel,count_loop):
    #L = L2                  # old method  >0.3
    #L = 0.5*R + 2*G - 1.5   # new method  >0.2+1.5
    #L = 0.4*R + 1.6*G - 1   # new method  >0.15+1.2
    L = 0.5*R + 2.5*G - 1.9  # new method  >0.2+1.9
    
    #  --- python3 function ---
    # input: L[*,*], R[*,*], G[*,*], maskW[*,*], maskA[*,*],maskS[*,*], camera ('Kiruna' or 'Abisko'), count = number of loop
    #        (this is output of H,L,S = RGB_everything_HLS(AllSkyPicture)
    #
    # output: RGBstrong,ASCindex = HLS_to_index(,,,,,)
    #           RGBstrong = jpg picture of strong aurora only
    #           ASCindex  = (cntMaybe, cntYes, cntStrong,
    #                        AverageLumS, expL_strong, LexpL_strong, LLL_strong)
    
    cntMaybe,cntYes,cntStrong = np.count_nonzero(maskW),np.count_nonzero(maskA),np.count_nonzero(maskS)
    
    Lstrong = np.zeros(L.shape,float) # Deine new arrays with the same dimension as H, S, L.
    Lstrong = L * maskS  # L values for auroral pixel, and zero for non-aurora pixels.
    
    #------------------------------------
    #SECTION (2) Calculate the activity of "Strong" auroral

    if camera == 'Kiruna' or camera == 'Kiruna0' or camera == 'Abisko':
        accumulate  = 2500 # about 1.5% of totalcount
        accumulate2 = 250
        accumulate3 = 25
        factor2 = min(math.sqrt(cntYes/(accumulate2*5)),1)
        factor  = min(math.sqrt(cntStrong/accumulate2),1) * factor2
    elif camera == 'Kiruna2' or  camera == 'Kiruna3':
        accumulate = 4900 # about 3% of totalcount
        accumulate2 = 1600
        accumulate3 = 50
        factor2 = min(math.sqrt(cntYes/(accumulate*10)),1)
        factor  = min(math.sqrt(cntStrong/accumulate2),1) * factor2

    Lsort           = np.sort(Lstrong, None)[-accumulate:]  # An array of the highest (most Luminous) L values of clear aurora.  The legth of the array is reduced to 3600 (=accumulate), i.e., less calculation time.  Here, dict['Factor'] = 1 for the first event of the night
    L_lowest, L_highest  = Lsort[0],Lsort[accumulate-1]   # minimum/maximum L value over 2500 pixels

    # normalize Lstrong with "fixed" vaues so that we can compare different filtering gives the same L
    if camera == 'Kiruna' or camera == 'Kiruna0' or camera == 'Abisko':
        Lsort    = (Lsort-0.1)/0.8
    elif camera == 'Kiruna2' or  camera == 'Kiruna3':
        Lsort    = (Lsort-0.2)/0.8

    Lsort = np.maximum(0,Lsort)
    LumStrongAccumulate = np.sum(Lsort) # sum(L) of the most Luminous 2500 pixels of clear aurora.
    # LumStrong           = np.sum(Lstrong) # summation of Luminocity values of all pixels of clear aurora
    expL_strong     = np.sum(np.exp(Lsort)) - accumulate  # summation of exp(L) of the most Luminous 2500 pixels of clear aurora.  Since exp(0) = 1, we must subtract numbers of pixels with "not clear aurora" = 2500-cntStrong
    LexpL_strong    = np.sum(Lsort*np.exp(Lsort))  # summation of Lexp(L) of the most Luminous 2500 pixels of clear aurora.
    LLL_strong      = (np.sum(Lsort*Lsort*Lsort))  # summation of L**3 of the most Luminous 2500 pixels of clear aurora.
    
    # Take averages --> divide by accumulate=2500 but not by cntStrong because size also matter
    #                   (scale of L is already adjusted from 0.1-0.9 to 0.0-1.0)

    if cntStrong >= accumulate:  # ignore if strong aurora does not cover thrshold pixels (2500)
        AverageLumS     = factor2*LumStrongAccumulate/accumulate # <L> in scale 0.1-0.9
        expL_strong     = factor2*expL_strong/accumulate
        LexpL_strong    = factor2*LexpL_strong/accumulate
        LLL_strong      = factor2*LLL_strong/accumulate # best to use
    elif (cntStrong >= accumulate2) | ( (cntStrong >= accumulate3) & (cntYes >= accumulate*2) ):
        AverageLumS     = factor2*LumStrongAccumulate/cntStrong # <L> in scale 0.1-0.9
        expL_strong     = factor2*expL_strong/cntStrong
        LexpL_strong    = factor2*LexpL_strong/cntStrong
        LLL_strong      = factor2*LLL_strong/cntStrong # best to use
    elif (cntStrong >= accumulate3) | ( (cntStrong >= accumulate3/5) & (cntYes >= accumulate2) ):  #  avoid artificial light
        AverageLumS     = factor*LumStrongAccumulate/cntStrong # <L> in scale 0.1-0.9
        expL_strong     = factor*expL_strong/cntStrong
        LexpL_strong    = factor*LexpL_strong/cntStrong
        LLL_strong      = factor*LLL_strong/cntStrong # best to use
    else:
        AverageLumS     = 0.
        expL_strong     = 0.
        LexpL_strong    = 0.
        LLL_strong      = 0.
    
    if math.isnan(AverageLumS) is True:     # This prevents the program to add nans which will raise errors at plotting.
        AverageLumS     = 0.0
        expL_strong     = 0.
        LexpL_strong    = 0.
        LLL_strong      = 0.

    return(cntMaybe,cntYes,cntStrong, AverageLumS, expL_strong, LexpL_strong, LLL_strong, L_lowest,L_highest)
