#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Oct 15 11:09:18 2020

@author: Peje
"""

import math
import pandas as pd
import numpy as np
import matplotlib.dates as mdates
from pathlib import Path
from datetime import datetime, timedelta, timezone
import os
import sys
import argparse
import logging
from aurora_detect import detect
import alerts
from read_aurora_index_csv import read_aurora_index_csv
from mkdelta import mkdelta

default_input_dir = '%Y/%m'
default_input_fname = Path("krn%Y%m%d_ASCindex.csv")
default_last_alert_fname = Path("last_alert.txt")
default_range = timedelta(minutes=10)


LOG_FORMAT = "%(asctime)s %(levelname) - %(message)s"

default_alerts = ['email', 'telegram']


def filename(args, start_time):
    outdir = Path(start_time.strftime(str(args.input_dir)))
    outname = outdir / Path(start_time.strftime(str(args.input_fname)))
    return outname


def alert(args, subject, msg, photoURL):
    for alert in args.alerts:
        if alert in alerts.available:
            alerts.available[alert](args, subject, msg, photoURL)
        else:
            logging.warning("unknown alert: {}".format(alert))


def main(args, show=False):
    start_time = args.end_time - args.range
    t = start_time
    files = []
    while t < args.end_time + timedelta(minutes=1):
        f = filename(args, t)
        if not files or f != files[-1]:
            logger.debug("appending: {}".format(f))
            files.append(filename(args, t))
        t += args.range
    frames = [read_aurora_index_csv(args, f) for f in files]
    res = []
    for val in frames:
        if not isinstance(val, type(None)):
            res.append(val)

    frames = res
    if not frames:
        return False

    result = pd.concat(frames, axis=0, ignore_index=True, sort=False)

    # filter out data outside start & end times
    result = result[(result['date_time'] >= start_time.replace(tzinfo=None))]
    result = result.loc[(result['date_time'] < args.end_time.replace(tzinfo=None))].reset_index()

    last_alert = datetime.strptime('1970 01 01 0000', '%Y %m %d %H%M')  # default
    try:
        with open(args.last_alert, 'r') as infile:
            for line in infile:
                if line.startswith('20'):
                    last_alert = datetime.strptime(line.strip(), '%Y-%m-%dT%H:%M:%S')
        logger.debug(f"last alert occurred at {last_alert}")
    except IOError as error:
        logger.info(f'oops, no {args.last_alert}')

    logger.debug(f"{last_alert}")

    if len(result) > 1:
        i = len(result)-1
        detection = detect(result, len(result)-1)
        time_since_last_alert = result['date_time'].iloc[i] - last_alert
        if detection > 0 and time_since_last_alert < timedelta(minutes=6):
            logger.debug(f"alert should be skipped {last_alert} {time_since_last_alert}")
        else:
            if detection == 100:
                alert(
                    args,
                    "Strong aurora is detected at Kiruna camera",
                    "<b>{}</b>\n{}".format(
                        result['date_time'].iloc[i].strftime("%F %H:%M UTC"),
                        "Strong aurora is detected at Kiruna camera, \n"
                        "see: <a href=\"https://www.irf.se/en/observatory-activities/allsky-camera/\">IRF Allsky-Camera</a>"
                    ),
                    result['date_time'].iloc[i].strftime(
                        "https://www.irf.se/alis/allsky/krn/%Y/%m/%d/%H/%FT%H.%M.00.000KRN_small.jpeg")
                )
                with open(args.last_alert, 'w') as outfile:
                    outfile.write(result['date_time'].iloc[i].strftime("%FT%T\n"))
            elif detection == 70:
                alert(
                    args,
                    "Strong aurora continues at Kiruna camera",
                    "<b>{}</b>\n{}".format(
                        result['date_time'].iloc[i].strftime("%F %H:%M UTC"),
                        "Strong aurora continues at Kiruna camera, \n"
                        "see: <a href=\"https://www.irf.se/en/observatory-activities/allsky-camera/\">IRF Allsky-Camera</a>"
                    ),
                    result['date_time'].iloc[i].strftime(
                        "https://www.irf.se/alis/allsky/krn/%Y/%m/%d/%H/%FT%H.%M.00.000KRN_small.jpeg")
                    )
                with open(args.last_alert, 'w') as outfile:
                    outfile.write(result['date_time'].iloc[i].strftime("%FT%T\n"))

    else:
        # Data files are loaded but not enough data to analyze after filter
        logger.info("{} - {} not enough data".format(start_time, args.end_time))

    secs = int(args.range.total_seconds())

    return True


if __name__ == '__main__':
    logging.basicConfig(format='%(asctime)s - %(levelname)s: %(message)s')
    logger = logging.getLogger()
    now = datetime.utcnow().replace(second=0, microsecond=0, tzinfo=timezone.utc)
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-log",
        "--log",
        default="warning",
        help=(
            "Provide logging level. "
            "Example --log debug', default='warning'"
        ),
    )

    group = parser.add_mutually_exclusive_group()
    group.add_argument('--range',
                       type=mkdelta,
                       default=timedelta(seconds=default_range.total_seconds()),
                       help="Process the last RANGE (default: {}m)".format(
                           default_range//timedelta(minutes=1)))
    parser.add_argument(
        "end_time", nargs=1,
        type=lambda s: datetime.strptime(s, '%Y-%m-%dT%H:%M:%S').replace(second=0,
                                                                         microsecond=0,
                                                                         tzinfo=timezone.utc),
        help="Stop processing at given time. Use format: %%Y-%%m-%%dT%%H:%%M:%%S "
             "(i.e: 2020-01-01T02:03:00)",
        default=now
        )
    parser.add_argument('--input-dir',
                        default=default_input_dir,
                        help="Specify where/how data is stored (default: %(default)s)")
    parser.add_argument('--input-fname', default=default_input_fname,
                        help="Specify input filename format INPUT_FNAME (default: %(default)s)")

    parser.add_argument('--last-alert', default=default_last_alert_fname,
                        help="File with time of last alert (default: %(default)s)")

    parser.add_argument(
        '-a', '--alert',  # either of this switches
        type=str,         # /parameters/ are ints
        dest='alerts',    # store in 'alerts'.
        default=[],       # since we're not specifying required.
        action='append',  # add to the list instead of replacing it
        help="One of: " + ', '.join(val for val in alerts.available) + ". You may specify multiple alerts."
    )
    args = parser.parse_args()
    if len(args.alerts) == 0:
        args.alerts = default_alerts

    if args.range < timedelta(minutes=6):
        logger.fatal("We need at least 6 minutes of data")
        exit(1)

    levels = {
        'critical': logging.CRITICAL,
        'error': logging.ERROR,
        'warn': logging.WARNING,
        'warning': logging.WARNING,
        'info': logging.INFO,
        'debug': logging.DEBUG
    }
    level = levels.get(args.log.lower())
    if level is None:
        raise ValueError(
            f"log level given: {args.log}"
            f" -- must be one of: {' | '.join(levels.keys())}")
    logger.setLevel(level=level)

    # When argparse uses the lambda we get a list of datetimes back, use the first
    if type(args.end_time) is list:
        args.end_time = args.end_time[0]

    logger.debug("range:       {}".format(args.range))
    logger.debug("end_time:    {}".format(args.end_time))
    logger.debug("input-dir:   {}".format(args.input_dir))
    logger.debug("input-fname: {}".format(args.input_fname))
    logger.debug("alerts:      {}".format(args.alerts))
    logger.debug("last-alert:  {}".format(args.last_alert))
    exit(not main(args))
