%function [Df,ht,dDl,d18l,vf_out,Tr,nstep] = evap_substep(Tclb,rhs,dDv_clb,dDva,d18v_clb,d18va,dzout,Nzout)


 clear all
 close all
 debug = true;

 % %e.g. input values to test the code
 Tclb=290; %cloud base temperature (K)
 rhs=.69; %Surface relative humidity from dropsonde
 dDv_clb=-73; %mean dD vapor isotope ratio at flight level
 d18v_clb=-11.7; %mean d18O vapor isotope ratio at flight level
 dDvapor_ambient=-70; %background dD vapor isotope ratio from nearest 150 m altitude
 d18vapor_ambient=-10.5; %background d18O vapor isotope ratio from nearest 150 m altitude

 zclb = 700; % default for Sarkar et al (2023, ACP submitted)
 Tsurf = 9.8e-3*zclb + Tclb; % (K) surface temperature, assuming 9.8 K/km lapse rate

 ZZ = [zclb 0];
 RH = [1 rhs]; %(fraction) linearly decreasing RH from cb to sf
 TT = [Tclb Tsurf]; %(K) Air temperature using lapse rate =9.8K/km
 DHDO = [dDvapor_ambient dDvapor_ambient];
 DH218O = [d18vapor_ambient d18vapor_ambient];

 Nzout=701; %Number of vertical levels from cloud base (700m) to surface
 dzout=1; %vertical spacing (deltaz=1)

ht=[zclb:-dzout:0]; %(m) Altitude from cloud base to surface
if Nzout ~= length(ht)
  disp('Warning: Mismatch between input Nzout and dzout for assumed 700m cloud base height.')
  disp(sprintf('  Proceeding with Nzout=%d',length(ht)))
  Nzout = length(ht);
end

% Diameter dimension
Dbins=[2.25,2.75,3.25,3.75,4.5,5.75,6.85,7.55,9.05,11.35,13.75,17.5,22.5,27.5,...
       32.5,37.5,42.5,47.5,75,125,175,225,275,325,375,500,700,900,1100,1300,...
       1500,1700,2000,2400,2800,3200,3600,4000,4400,4800,5500,6500,7500,8500,...
       9500,12500,17500,22500,27500]'.*1e-6;%(m), bins from CIP+PIP stitched
Ndiam = length(Dbins);

% initialize arrays for output
Df = NaN(Ndiam,Nzout);
Tr = NaN(Ndiam,Nzout);
dDl = NaN(Ndiam,Nzout);
d18l = NaN(Ndiam,Nzout);
vf_out = NaN(Ndiam,Nzout);
Fh_out = NaN(Ndiam,Nzout);
Fv_out = NaN(Ndiam,Nzout);
nstep = zeros(Ndiam,1);

%Obtaining cloud base liquid isotope ratios
% Initialize droplet properties at cloud base before integrating downwards
% Assume rain drops in equilibrium with vapor at cloud base
Df(:,1) = Dbins;
Tr(:,1) = Tclb; % rain temperature same as air temperature.
dDl(:,1) = (alphaD(Tclb).*(dDv_clb+1e3))-1e3; % isotopic equilibrium with observed vapor%
d18l(:,1) = (alpha18(Tclb).*(d18v_clb+1e3))-1e3; % isotopic equilibrium with observed vapor%

Df(1:19,1)=NaN; Df(43:end,1)=NaN; % unusable bins: outside of observed size range

L=2.25e6;%(J/kg) Latent heat of vaporization
Rv=461;%specific gas constant for water vapor (J/kg/K)
cw=4187;%specific heat of liq water (J/Kg/K)
rhol=1e3;%J/kg, density of water

% integrate each diameter bin separately because step size restrictions vary
% with 1/D^2
for i = 21:4:40 %1:Ndiam
  
  % initialize array and counter to keep track of last 1000 steps
  %   Useful to confirm that integration is stable.
  last1000steps = NaN(6,1000); 
  ilast1000 = 1;

  for j=2:Nzout

    if isnan(Df(i,j-1));
      % do not perform integration when diameter is set to NaN
      % as in unused bins or after full evaporation      
      continue
    end
    
    % integrate down from ht(j-1) to ht(j), adapting the step size if needed.
    z = ht(j-1);
    dzout = ht(j-1) - ht(j);
    dz_remaining = dzout;
    
    % copy present values to next level below.
    % they will serve as the initial condition for the integration    
    Df(i,j) = Df(i,j-1);
    Tr(i,j) = Tr(i,j-1);
    dDl(i,j) = dDl(i,j-1);
    d18l(i,j) = d18l(i,j-1);
    
    while dz_remaining > 0
      
      rh = interp1(ZZ,RH,z);
      Tas = interp1(ZZ,TT,z);
      dDva = interp1(ZZ,DHDO,z);
      d18va = interp1(ZZ,DH218O,z);

      esa=es(Tas);
      pa = 101325.*((1-(2.25577e-5.*z)).^5.2559); %Air pressure(Pa) Salamalikis et al. (2019)
      rhoa = (pa-(0.378.*rh.*esa))./(287*Tas); %(kg/m3) Air density, (Salamalikis et al. 2019)

      vf = terminal_velocity_rain(Df(i,j),rhoa); %(m/s) Terminal velocity, +dn,-up, Graf 2017, eq. 3.10
      Dva = diffusivity_water_vapor_in_air(Tas,pa); %(m2s-1) Diffusivity of vapor, Graf 2017, eq. C3.1
      ka= thermal_conductivity_air(Tas); %(Jm-1s-1K-1) Thermal conductivity of moist air, Graf 2017...
      Fh = ventilation_coefficient_heat(vf,Df(i,j),Tas,rhoa);
      Fv = ventilation_coefficient_mass(vf,Df(i,j),Tas,rhoa,pa);

      % Compute fractionation factor based on ambient temperature
      % TODO: test whether using rain temperature changes results.
      Tfrac = Tas; % Tfrac = Tr(i,j); % 
      alphaDvl=alphaD(Tfrac); %Majoube (1971b)
      alpha18vl=alpha18(Tfrac); %Majoube (1971b)

      % diffusivity ratio (heavy to light in air), 
      Dva_ratio_D = 0.9755; Dva_ratio_18 = 0.9723; % Merlivat (??)
%      Dva_ratio_D = 0.9839; Dva_ratio_18 = 0.9691; % Cappa et al (2003)
%      Dva_ratio_D = 0.9755; Dva_ratio_18 = 0.9710; % Hellman & Harvey (2020, GRL)

      % estimate maximum step size based on leading coefficient of d(Tr)/dz equation
      dzmax = ( Df(i,j).^2.*vf*rhol*cw ) / ( 12.*Fh.*ka );

      % step size for this substep
      dz = min( dz_remaining, 0.5*dzmax );

      % update es(Train) and compute useful quantities
      esr = es(Tr(i,j));
      x = esr./Tr(i,j) - (rh*esa)./Tas;
      y = Tr(i,j) - Tas;
      
      %dD/dz obtained from Salamalikis et al. 2016, 
      %dTr/dz obtained from Graf et al. 2019
      dDdz = - (4.*Fv.*Dva.*x) ./ (Df(i,j).*vf.*rhol.*Rv);
      dTdz = - ( (12.*Fh.*ka) ./ (Df(i,j).^2.*vf.*rhol.*cw) ) ...
             .* ( y + ( (Dva.*L.*x.*Fv) ./ (Fh.*ka.*Rv) ) );
      
      %dDl/dz & d18Ol/dz obtained from Salamalikis et al. 2016 with the
      %correction using '12' instead of '3'
      ddz_dDl = ( 12.*esr.*Fv.*Dva ./ (rhol.*Rv.*(Df(i,j).^2).*Tr(i,j).*vf) ) ...
                .*( Dva_ratio_D^.58 .* ( (dDva+1e3).*rh.*esa.*Tr(i,j) ./ (esr.*Tas)  ... 
                                         - ( dDl(i,j)+1e3)./alphaDvl ) ...
                    - (dDl(i,j)+1e3) .* ( rh*esa*Tr(i,j)./(esr*Tas) - 1 ) ); % 

      ddz_d18l = ( 12.*esr.*Fv.*Dva ./ (rhol.*Rv.*(Df(i,j).^2).*Tr(i,j).*vf) ) ...
          .*( Dva_ratio_18^.58 .* ( (d18va+1e3).*rh.*esa.*Tr(i,j) ./ (esr.*Tas)  ...
                                    - ( d18l(i,j)+1e3)./alpha18vl ) ...
              - (d18l(i,j)+1e3) .* ( rh*esa*Tr(i,j)./(esr*Tas) - 1 ) );
      %      ddz_dDl = ( 12.*esr.*Fv.*Dva ./ (rhol.*Rv.*(Df(i,j).^2).*Tr(i,j).*vf) ) ...
      %                   .*( Dva_ratio_D^.58 .* ( (dDva+1e3).*rh.*esa.*Tr(i,j) ./ (esr.*Tas)  ... % .*esa.*Tas ./ (esr.*Tr(i,j))  ...
      %					    - ( dDl(i,j)+1e3)./alphaDvl ) ...
      %		     - (dDl(i,j)+1e3) .* ( rh*esa*Tas./(esr*Tr(i,j)) - 1 ) ); % 
      %
      %      ddz_d18l = ( 12.*esr.*Fv.*Dva ./ (rhol.*Rv.*(Df(i,j).^2).*Tr(i,j).*vf) ) ...
      %		.*( Dva_ratio_18^.58 .* ( (dDva+1e3).*rh.*esa.*Tas ./ (esr.*Tr(i,j))  ...
      %					    - ( d18l(i,j)+1e3)./alpha18vl ) ...
      %		     - (d18l(i,j)+1e3) .* ( rh*esa*Tas./(esr*Tr(i,j)) - 1 ) );
      %
      %    d18l(i,j)=d18l(i,j)+(((12.*esa.*Fv.*Dva./(rhol.*Rv.*(Df(i,j).^2).*Tr(i,j).*vf)).*...
      %    (((Dva_ratio_18^.58).*(((d18va+1e3).*rh.*(esa.*Tas./(esr.*Tr(i,j))))-((d18l(i,j)+1e3)./alpha18vl(1,j))))...
      %    -((d18l(i,j)+1e3).*((rh.*((esa.*Tas)./(esr.*Tr(i,j))))-1)))).*dz);        

      % increment height and predicted variables.
      z = z - dz;
      Df(i,j) = Df(i,j) + dz*dDdz;
      Tr(i,j) = Tr(i,j) + dz*dTdz;
      dDl(i,j) = dDl(i,j) + dz*ddz_dDl;
      d18l(i,j) = d18l(i,j) + dz*ddz_d18l;

      % update remaining height to integrate between ht(i-1) and ht(i).
      dz_remaining = dz_remaining - dz;
      nstep(i) = nstep(i) + 1;
      
      if Df(i,j) < 1e-6
        % NaN all values for drops smaller than 1 micron
        Df(i,j) = NaN;
        Tr(i,j) = NaN;
        dDl(i,j) = NaN;
        d18l(i,j) = NaN;
        vf = NaN;
      end

      % save terminal velocity for output.
      vf_out(i,j) = vf; % save terminal velocity
      Fh_out(i,j) = Fh; % save terminal velocity
      Fv_out(i,j) = Fv; % save terminal velocity

      
      if ilast1000 > 1000; ilast1000 = 1; end
      last1000steps(:,ilast1000) = [z; Df(i,j); Tr(i,j); dDl(i,j); ...
                          d18l(i,j); vf];
      ilast1000 = ilast1000 + 1;

    end % while dz_remaining > 0
    
  end % for j = 2:Nzout

  disp(sprintf('D0 = %8.2e m, Nstep = %d, mean dz = %3g m', ...
               Dbins(i), nstep(i), zclb/nstep(i)));

  if debug
    last1000steps = circshift(last1000steps,1000+1-ilast1000,2);

    figure; 
    subplot(2,5,1); plot(Df(i,:),ht,last1000steps(2,:), ...
                         last1000steps(1,:),'o-'); xlabel('D, m')
    subplot(2,5,2); plot(Tr(i,:),ht,last1000steps(3,:), ...
                         last1000steps(1,:),'o-'); xlabel('Tr, K')
    subplot(2,5,3); plot(dDl(i,:),ht,last1000steps(4,:), ...
                         last1000steps(1,:),'o-'); xlabel('dD rain, per mil')
    subplot(2,5,4); plot(d18l(i,:),ht,last1000steps(5,:), ...
                         last1000steps(1,:),'o-'); ...
        xlabel('d18O rain, per mil')
    subplot(2,5,5); plot(dDl(i,:)-8*d18l(i,:),ht,last1000steps(4,:)-8*last1000steps(5,:),last1000steps(1,:),'o-'); ...
        xlabel('dex rain, per mil')
    subplot(2,5,6); plot(vf_out(i,:),ht,last1000steps(6,:),last1000steps(1,:),'o-'); ...
        xlabel('vt rain, m/s')

    rh_over_rain = interp1(ZZ,RH,ht).*es(interp1(ZZ,TT,ht))./es(Tr(i,:));

    subplot(2,5,7); semilogx(1-interp1(ZZ,RH,ht),ht, ...
                         1-rh_over_rain,ht,'--'); ...
                         xlabel('1-RH of air, wrt Train');
    subplot(2,5,8); plot(interp1(ZZ,TT,ht),ht,Tr(i,:),ht,'--'); ...
                         xlabel('Tair, Train');
    subplot(2,5,9); plot(Fh_out(i,:),ht); ...
                         xlabel('Fh');
    subplot(2,5,10); plot(Fv_out(i,:),ht); ...
                         xlabel('Fv');

    figure
    plot(rh_over_rain,dDl(i,:)-8*d18l(i,:),'o-', ...
         rh_over_rain,interp1(ZZ,DHDO,ht)-8*interp1(ZZ,DH218O,ht));
    xlabel('rh over rain'); ylabel('d excess rain')

% $$$     subplot(2,5,9); plot(interp1(ZZ,DHDO,ht),ht); ...
% $$$                          xlabel('dDv ambient');
% $$$     subplot(2,5,10); plot(interp1(ZZ,DH218O,ht),ht); ...
% $$$                          xlabel('d18Ov ambient');

    input('Hit return to continue')
  end

end % for i = 1:Ndiam


